package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import javax.annotation.Nullable;
import java.util.Objects;

/**
 * @author Nikolay Firov <a href="mailto:firov@yandex-team.ru"></a>
 * @date 24.04.18
 */
public class LogbrokerSource {
    private final String ident;
    private final String logType;

    public LogbrokerSource(String id) {
        this(id, null);
    }

    public LogbrokerSource(String id, @Nullable String logType) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(id));

        this.ident = id;
        this.logType = logType;
    }

    public boolean includes(LogbrokerSource source) {
        if (getLogType() == null) {
            return getIdent().equals(source.getIdent());
        }

        return equals(source);
    }

    public static LogbrokerSource fromSourcePath(String path) {
        String[] parts = path.split("--");
        Preconditions.checkState(parts.length == 1 || parts.length == 2);
        return new LogbrokerSource(parts[0], (parts.length > 1) ? parts[1] : null);
    }

    /**
     * Парсит полное имя топика, например rt3.iva--other--statbox-synthetic-log.
     *
     * @param topic
     * @return
     */
    public static LogbrokerSource fromTopic(String topic) {
        String[] parts = topic.split("--");
        Preconditions.checkArgument(
            parts.length == 3,
            "Full topic name (%s) must be split exactly into 3 parts, got %s", topic, parts.length
        );
        return new LogbrokerSource(parts[1], parts[2]);
    }

    /**
     * Парсит полное имя партиции, например rt3.iva--other--statbox-synthetic-log:42.
     *
     * @param partition
     * @return
     */
    public static LogbrokerSource fromPartition(String partition) {
        int indexOfColon = partition.lastIndexOf(':');
        Preconditions.checkArgument(
            indexOfColon >= 0, "Unable to find ':' in partition " + partition
        );
        return fromTopic(partition.substring(0, indexOfColon));
    }

    @Override
    public String toString() {
        return "LogbrokerSource{" +
            "ident='" + getIdent() + '\'' +
            ", logType='" + getLogType() + '\'' +
            '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        LogbrokerSource topic = (LogbrokerSource) o;
        return Objects.equals(getIdent(), topic.getIdent()) && Objects.equals(getLogType(), topic.getLogType());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getIdent(), getLogType());
    }

    public String getIdent() {
        return ident;
    }

    /**
     * Если null, читать все logType'ы.
     */
    public String getLogType() {
        return logType;
    }
}
