package ru.yandex.market.logshatter.reader.logbroker;

import ru.yandex.market.logbroker.pull.LogBrokerSourceKey;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.SourceContext;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/07/15
 */
public class LogbrokerSourceContext extends SourceContext {

    private final LogBrokerSourceKey logBrokerSourceKey;
    private final Path path;
    private final int instanceId;

    // последний обработанный
    private volatile long partitionOffset;
    private volatile long seqno;
    // последний вычитаннный
    private volatile long readerPartitionOffset;
    private volatile long readerSeqno;

    public LogbrokerSourceContext(Path path, LogShatterConfig logShatterConfig, LogBrokerSourceKey sourceKey,
                                  BatchErrorLoggerFactory errorLoggerFactory, int instanceId,
                                  ReadSemaphore.QueuesCounter counter) {
        super(logShatterConfig, sourceKey.getIdent(), sourceKey.getId(), errorLoggerFactory, counter);
        this.logBrokerSourceKey = sourceKey;
        this.path = path;
        this.instanceId = instanceId;
    }

    @Override
    public long getDataOffset() {
        return partitionOffset;
    }

    @Override
    public void setDataOffset(long dataOffset) {
        partitionOffset = Math.max(partitionOffset, dataOffset);
    }


    @Override
    public void setFileOffset(long fileOffset) {
        seqno = Math.max(seqno, fileOffset);
    }


    @Override
    public long getFileOffset() {
        return seqno;
    }

    public long getReaderPartitionOffset() {
        return Math.max(readerPartitionOffset, partitionOffset);
    }

    public void setReaderPartitionOffset(long readerPartitionOffset) {
        this.readerPartitionOffset = readerPartitionOffset;
    }

    public long getReaderSeqno() {
        return Math.max(readerSeqno, seqno);
    }

    public void setReaderSeqno(long readerSeqno) {
        this.readerSeqno = readerSeqno;
    }

    public long getPartitionOffset() {
        return partitionOffset;
    }

    public long getSeqno() {
        return seqno;
    }

    @Override
    public String getName() {
        return logBrokerSourceKey.getPath();
    }

    @Override
    public int getInstanceId() {
        return instanceId;
    }

    @Override
    public String getLogBrokerTopic() {
        return logBrokerSourceKey.getTopic();
    }

    @Override
    public Path getPath() {
        return path;
    }

    @Override
    public boolean isClosed() {
        return false; //TODO
    }

    @Override
    public String getHost() {
        return logBrokerSourceKey.getServer();
    }

    @Override
    public synchronized CompletableFuture<Void> finish() {
        this.readerSeqno = 0;
        return super.finish();
    }

    @Override
    public String toString() {
        return "LogbrokerSourceContext{" +
            "logBrokerSourceKey=" + logBrokerSourceKey +
            ", path=" + path +
            '}';
    }
}
