package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.base.Stopwatch;
import org.apache.commons.io.IOUtils;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logbroker.pull.LogBrokerSession;
import ru.yandex.market.logbroker.pull.MessageMeta;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.LogShatterService;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.logshatter.reader.QueuesLimits;
import ru.yandex.market.logshatter.reader.ReadSemaphore;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 30/07/15
 */
public class PartitionContext implements LeaderLatchListener {

    private static final Logger log = LogManager.getLogger();

    /**
     * Полное имя партиции: topic:partition.
     */
    private final String name;

    private final PartitionManager partitionManager;
    private final LogbrokerSource source;
    private final ReadSemaphore.QueuesCounter queuesCounter;
    private final ReadSemaphore readSemaphore;
    private final LogShatterService logShatterService;

    /**
     * Хост-лидер партиции, с которого должно происходить чтение.
     */
    private volatile String host;
    private volatile long lag;
    private volatile long logStart;
    private volatile long logEnd;
    private volatile String owner;
    private volatile LogBrokerSession session;
    private volatile LeaderLatch leaderLatch;
    /**
     * Набор еще не обработанных батчей. Используется для проверки все ли батчи обработаны и можно ли коммитить offset.
     */
    private final Queue<MessageOffset> offsets = new ArrayDeque<>();
    /**
     * Последний обработанный оффсет, с которым можем сделать hard commit.
     */
    private volatile long committableOffset = -1;
    /**
     * Последний оффсет, записанный в хранилище метаданных LB-кластера при hard commit-е.
     */
    private volatile long lastCommitedOffset = -1;
    /**
     * Последний вычитанный номер сообщения в партиции, коммитим его, чтобы не читать одно и то же.
     */
    private volatile long softOffset = -1;
    private boolean needReload = true;

    private final LogBrokerPartitionSourceContextsStorage sourceContextsStorage;

    PartitionContext(
        LogBrokerOffset offset,
        LogbrokerSource source,
        ReadSemaphore.QueuesCounter queuesCounter,
        PartitionManager partitionManager,
        LogBrokerPartitionSourceContextsStorage sourceContextsStorage,
        ReadSemaphore readSemaphore,
        LogShatterService logShatterService
    ) {
        this.name = offset.getPartition();
        this.source = source;
        this.queuesCounter = queuesCounter;
        this.partitionManager = partitionManager;
        this.sourceContextsStorage = sourceContextsStorage;
        this.readSemaphore = readSemaphore;
        this.logShatterService = logShatterService;
        update(offset);
    }

    public void update(LogBrokerOffset offset) {
        lag = offset.getLag();
        logStart = offset.getLogStart();
        logEnd = offset.getLogEnd();
        owner = offset.getOwner();
    }

    public boolean hasSession() {
        return session != null;
    }

    public void closeSession() {
        if (hasSession()) {
            try {
                session.close();
            } catch (IOException e) {
                log.error("Exception while closing session", e);
            }
            session = null;
        }
    }

    public boolean isReadAllowed() {
        if (!leaderLatch.hasLeadership()) {
            return false;
        }

        QueuesLimits.QueueLimit limit = queuesCounter.getQueueThatReachedLimit();
        if (limit != null) {
            log.info(
                "Limit for queue {} with rule {} exceeded. Read disallowed for partition: {} ",
                limit.getQueueId(), limit.getRule().pattern(), name
            );
            return false;
        }

        return true;
    }

    public void addDataToParseQueues(MessageMeta meta, InputStream inputStream) throws InterruptedException, IOException {
        readSemaphore.waitForRead();

        Stopwatch stopwatch = Stopwatch.createStarted();

        readSemaphore.notifyRead();

        List<LogbrokerSourceContext> sourceContexts =
            sourceContextsStorage.getSourceContexts(meta.getLogBrokerSourceKey(), meta.getInstanceId()).stream()
                .filter(sourceContext -> {
                    // повторно читаем уже обработанный чанк, поэтому пропускаем
                    return meta.getSeqno() > sourceContext.getReaderSeqno();
                })
                .collect(Collectors.toList());

        addOffset(meta, sourceContexts);
        if (sourceContexts.isEmpty()) {
            return;
        }

        List<String> lines = IOUtils.readLines(inputStream, StandardCharsets.UTF_8);

        Duration readDuration = stopwatch.elapsed();

        long batchSizeBytes = LogShatterUtil.calcSizeBytes(lines);

        for (LogbrokerSourceContext sourceContext : sourceContexts) {
            sourceContext.setReaderSeqno(meta.getSeqno());
            sourceContext.setReaderPartitionOffset(meta.getOffset());

            LogBatch logBatch = new LogBatch(
                lines.stream(),
                meta.getOffset(),
                meta.getSeqno(),
                batchSizeBytes,
                readDuration,
                sourceContext.getLogParser().getTableDescription().getColumns(),
                name
            );

            queuesCounter.increment(batchSizeBytes);
            readSemaphore.incrementGlobalQueue(batchSizeBytes);

            sourceContext.getParseQueue().add(logBatch);
            logShatterService.addToParseQueue(sourceContext);
        }
    }

    private synchronized void addOffset(MessageMeta meta, List<LogbrokerSourceContext> sourceContexts) {
        softOffset = meta.getOffset();
        checkOffsets();
        if (sourceContexts.isEmpty() && offsets.isEmpty()) { // обработчиков для этих данных нет, поэтому пропускаем
            committableOffset = meta.getOffset();
            return;
        }
        offsets.add(new MessageOffset(meta.getOffset(), meta.getSeqno(), sourceContexts));
    }

    /**
     * Если чанк обработан всеми парсерами, то сохраняем offset для коммита и удаляем его из очереди.
     * Идём до 1ого не обработанного чанка.
     * Тут важно, что чанки читались и писались в очередь последовательно, а обрабатывались асинхронно. Коммитим, когда
     * все чанки до этого уже обработаны.
     */
    private synchronized void checkOffsets() {
        while (!offsets.isEmpty()) {
            MessageOffset messageOffset = offsets.element();
            if (!messageOffset.isProcessed()) {
                break;
            }
            committableOffset = messageOffset.getPartitionOffset();
            offsets.remove();
        }
    }

    public synchronized void clearOffsetsState() {
        committableOffset = 0;
        offsets.clear();
    }

    public long getCommittableOffset() {
        checkOffsets();
        return committableOffset;
    }

    public void setCommittableOffset(long committableOffset) {
        this.committableOffset = committableOffset;
    }

    public long getLastCommitedOffset() {
        return lastCommitedOffset;
    }

    public long getSoftOffset() {
        return softOffset;
    }

    public void setLastCommitedOffset(long lastCommitedOffset) {
        this.lastCommitedOffset = lastCommitedOffset;
    }

    @Override
    public void isLeader() {
        needReload = true;
        partitionManager.onLeader(this);
    }

    @Override
    public void notLeader() {
        partitionManager.notLeader(this);
    }


    public void maybeReloadSourceContexts() {
        if (needReload) {
            sourceContextsStorage.reloadAll();
            needReload = false;
        }
    }

    public LeaderLatch getLeaderLatch() {
        return leaderLatch;
    }

    public void setLeaderLatch(LeaderLatch leaderLatch) {
        this.leaderLatch = leaderLatch;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public void setLag(long lag) {
        this.lag = lag;
    }

    public String getHost() {
        return host;
    }

    public long getLag() {
        return lag;
    }

    public String getName() {
        return name;
    }

    public long getLogStart() {
        return logStart;
    }

    public long getLogEnd() {
        return logEnd;
    }

    public String getOwner() {
        return owner;
    }

    public void setSession(LogBrokerSession session) {
        this.session = session;
    }

    public LogBrokerSession getSession() {
        return session;
    }

    public CompletableFuture<Void> release() {
        return sourceContextsStorage.finishAll();
    }

    public LogbrokerSource getSource() {
        return source;
    }

    public ReadSemaphore.QueuesCounter getQueuesCounter() {
        return queuesCounter;
    }
}
