package ru.yandex.market.logshatter.reader.logbroker;

import com.mongodb.MongoClient;
import com.mongodb.bulk.BulkWriteResult;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.UpdateManyModel;
import com.mongodb.client.model.UpdateOptions;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bson.BsonReader;
import org.bson.BsonWriter;
import org.bson.Document;
import org.bson.codecs.Codec;
import org.bson.codecs.DecoderContext;
import org.bson.codecs.DocumentCodec;
import org.bson.codecs.EncoderContext;
import org.bson.codecs.configuration.CodecRegistries;
import org.bson.codecs.configuration.CodecRegistry;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author kukabara
 */
public class PartitionDao {
    private static final Logger log = LogManager.getLogger();

    private static final UpdateOptions UPSERT = new UpdateOptions().upsert(true);
    private static final CodecRegistry CODEC_REGISTRY = CodecRegistries.fromRegistries(
        MongoClient.getDefaultCodecRegistry(),
        CodecRegistries.fromCodecs(new LogBrokerOffsetCodec())
    );

    private final MongoCollection<LogBrokerOffset> collection;

    public PartitionDao(MongoDatabase database) {
        collection = database
            .withCodecRegistry(CODEC_REGISTRY)
            .getCollection("partition", LogBrokerOffset.class);
    }


    private static class LogBrokerOffsetCodec implements Codec<LogBrokerOffset> {
        private static final Codec<Document> DOCUMENT_CODEC = new DocumentCodec();

        @Override
        public LogBrokerOffset decode(BsonReader reader, DecoderContext decoderContext) {
            Document document = DOCUMENT_CODEC.decode(reader, decoderContext);
            return new LogBrokerOffset(
                document.getString("partition"),
                document.getLong("offset"),
                document.getLong("logStart"),
                document.getLong("logEnd"),
                document.getLong("lag"),
                "",
                document.getString("dc")
            );
        }

        @Override
        public void encode(BsonWriter writer, LogBrokerOffset offset, EncoderContext encoderContext) {
            Document document = new Document();
            document.put("partition", offset.getPartition());
            document.put("offset", offset.getOffset());
            document.put("logStart", offset.getLogStart());
            document.put("logEnd", offset.getLogEnd());
            document.put("lag", offset.getLag());
            document.put("dc", offset.getDc());
            document.put("updated", new Date());
            DOCUMENT_CODEC.encode(writer, document, encoderContext);
        }

        @Override
        public Class<LogBrokerOffset> getEncoderClass() {
            return LogBrokerOffset.class;
        }
    }

    private static Document toIdDocument(LogBrokerOffset offset) {
        return toIdDocument(offset.getPartition());
    }

    private static Document toIdDocument(String partition) {
        return new Document("_id", partition);
    }

    public LogBrokerOffset get(LogBrokerOffset offset) {
        return get(offset.getPartition());
    }

    /**
     * @param partition "rt3.fol--market-health-stable--other:28"
     */
    public LogBrokerOffset get(String partition) {
        for (LogBrokerOffset offset : collection.find(toIdDocument(partition))) {
            return offset;
        }
        return null;
    }

    public List<LogBrokerOffset> getAll() {
        List<LogBrokerOffset> offsets = new ArrayList<>();
        for (LogBrokerOffset offset : collection.find()) {
            offsets.add(offset);
        }
        return offsets;
    }

    public boolean delete(String partition) {
        return collection.deleteOne(toIdDocument(partition)).getDeletedCount() > 0;
    }

    public void save(LogBrokerOffset offset) {
        collection.updateOne(toIdDocument(offset), new Document("$set", offset), UPSERT);
    }

    /**
     * Сохраняем оффсеты, полученные от одного из ДЦ Логброкера в Монгу так, чтобы в Монге получалась общая картина лага
     * по партициям с учётом того, что мы иногда читаем отзеркалированные данные.
     */
    public void advanceOffsets(Collection<LogBrokerOffset> offsets) {
        log.info("advanceOffsets saving {} offsets", offsets.size());

        if (offsets.isEmpty()) {
            return;
        }

        BulkWriteResult result = collection.bulkWrite(
            offsets.stream()
                .flatMap(offset -> Stream.of(
                    new UpdateManyModel<LogBrokerOffset>(
                        Filters.or(
                            // В Логброкере оффсет больше чем в Монге, это значит что в Логброкере более свежая
                            // информация об оффсетах, и нужно её сохранить в Монгу. Если оффсет больше в Монге, то
                            // значит более свежая информация об оффсетах была в другом ДЦ Логброкера, и мы её уже
                            // сохранили в Монгу. Если оффсеты в Монге и в Логброкере равны, то надо смотреть на logEnd.
                            Filters.and(
                                toIdDocument(offset),
                                Filters.lt("offset", offset.getOffset())
                            ),
                            // Оффсеты в Логброкере и Монге совпадают, но logEnd (он же logSize) в Логброкере больше или
                            // равен logEnd в Монге. Это значит что Логшаттер ничего не прочитал с момента последнего
                            // сохранения оффсетов, но в Логброкере появились новые данные, и лаг увеличился. Это надо
                            // записать в Монгу. Если logEnd в Монге и в Логброкере равны, то всё равно записываем на
                            // всякий случай.
                            Filters.and(
                                toIdDocument(offset),
                                Filters.eq("offset", offset.getOffset()),
                                Filters.lte("logEnd", offset.getLogEnd())
                            )
                        ),
                        new Document("$set", offset)
                    ),
                    // В Монге ещё нет оффсета для этой партиции
                    new UpdateManyModel<LogBrokerOffset>(
                        toIdDocument(offset),
                        new Document("$setOnInsert", offset),
                        UPSERT
                    )
                ))
                .collect(Collectors.toList())
        );

        log.info(
            "Saved {} offsets. Deleted {}, inserted {}, matched {}, modified {}.",
            offsets.size(),
            result.getDeletedCount(), result.getInsertedCount(), result.getMatchedCount(), result.getModifiedCount()
        );
    }
}
