package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;
import com.google.common.collect.Sets;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.zookeeper.KeeperException;
import ru.yandex.market.health.ProcessingQueue;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logbroker.pull.LogBrokerPartition;
import ru.yandex.market.logshatter.LogShatterService;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.meta.LogshatterMetaDao;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerSourcesWithoutMetadataMonitoring;

import javax.annotation.concurrent.ThreadSafe;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 27/03/2017
 */
@ThreadSafe
public class PartitionManager {

    private static final Logger log = LogManager.getLogger();

    private final LogShatterService logShatterService;
    private final LogBrokerConfigurationService logBrokerConfigurationService;
    private final ReadSemaphore readSemaphore;
    private final LogshatterMetaDao logshatterMetaDao;
    private final LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring;
    private final BatchErrorLoggerFactory errorLoggerFactory;
    private final int maxAllowedPartitions;
    private final int partitionsPerReleaseAttempt;
    private int lockAcquireTimeMillis = 1000;

    private final Set<String> activeLocks = Sets.newConcurrentHashSet();
    private final ConcurrentMap<String, PartitionContext> partitionContexts = new ConcurrentHashMap<>();
    private final int clickhouseSocketTimeoutSeconds;

    private final ProcessingQueue<PartitionContext> partitionQueue = new ProcessingQueue<PartitionContext>() {
        @Override
        protected boolean isUpdated(PartitionContext partitionContext) {
            return true;
        }
    };

    public PartitionManager(
        LogShatterService logShatterService, LogBrokerConfigurationService logBrokerConfigurationService,
        ReadSemaphore readSemaphore,
        LogshatterMetaDao logshatterMetaDao,
        LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring,
        BatchErrorLoggerFactory errorLoggerFactory,
        int maxAllowedPartitions,
        int partitionsPerReleaseAttempt,
        int clickhouseSocketTimeoutSeconds
    ) {
        this.logShatterService = logShatterService;
        this.logBrokerConfigurationService = logBrokerConfigurationService;
        this.readSemaphore = readSemaphore;
        this.logshatterMetaDao = logshatterMetaDao;
        this.sourcesWithoutMetadataMonitoring = sourcesWithoutMetadataMonitoring;
        this.errorLoggerFactory = errorLoggerFactory;
        this.maxAllowedPartitions = maxAllowedPartitions;
        this.partitionsPerReleaseAttempt = partitionsPerReleaseAttempt;
        this.clickhouseSocketTimeoutSeconds = clickhouseSocketTimeoutSeconds;
    }

    public void onLeader(PartitionContext partitionContext) {
        activeLocks.add(partitionContext.getName());
        releaseExtraLocks();
    }

    public void notLeader(PartitionContext partitionContext) {
        partitionContext.release();
        activeLocks.remove(partitionContext.getName());
    }

    public void releaseExtraLocks() {
        if (maxAllowedPartitions <= 0) {
            return;
        }
        for (int i = 0; i < partitionsPerReleaseAttempt && activeLocks.size() > maxAllowedPartitions; i++) {
            PartitionContext context = partitionQueue.maybeTakeLock();
            if (context == null) {
                continue;
            }
            try {
                if (!context.getLeaderLatch().hasLeadership()) {
                    continue;
                }
                log.info(
                    "Too many locks current {}, max {}). Releasing lock for {}",
                    activeLocks.size(), maxAllowedPartitions, context.getName()
                );
                releaseLock(context);
                log.info("Released lock for {}", context.getName());
            } finally {
                partitionQueue.returnLock(context);
            }
        }
    }

    public void updatePartitions(List<LogBrokerOffset> offsets, List<LogBrokerPartition> partitions) {
        for (LogBrokerOffset offset : offsets) {
            PartitionContext context = partitionContexts.computeIfAbsent(
                offset.getPartition(),
                s -> {
                    LogbrokerSource source = LogbrokerSource.fromPartition(offset.getPartition());

                    PartitionContext partitionContext = new PartitionContext(
                        offset,
                        source,
                        readSemaphore.getQueuesCounterForSource(
                            String.format("%s--%s", source.getIdent(), source.getLogType())
                        ),
                        this,
                        new LogBrokerPartitionSourceContextsStorage(
                            logBrokerConfigurationService,
                            readSemaphore,
                            logshatterMetaDao,
                            sourcesWithoutMetadataMonitoring,
                            errorLoggerFactory
                        ),
                        readSemaphore,
                        logShatterService
                    );

                    try {
                        createLeaderLatch(partitionContext);
                    } catch (Exception e) {
                        log.info("Failed to create Leader latch for partition", e);
                    }
                    return partitionContext;
                }
            );
            context.update(offset);
        }

        for (LogBrokerPartition partition : partitions) {
            PartitionContext partitionContext = partitionContexts.get(partition.getName());
            if (partitionContext == null) {
                log.error("Unknown partition: " + partition.toString());
                continue;
            }
            if (partitionContext.getHost() == null) {
                partitionContext.setHost(partition.getHost());
                partitionQueue.add(partitionContext);
            } else if (!partitionContext.getHost().equals(partition.getHost())) {
                log.info(
                    "Partition {} leader host changed. Old: {}, new: {}",
                    partition.getName(), partitionContext.getHost(), partition.getHost()
                );
                partitionContext.setHost(partition.getHost());
            }
        }
    }

    public PartitionContext acquirePartition() throws Exception {
        while (true) {
            PartitionContext partitionContext = partitionQueue.takeLock();
            try {
                LeaderLatch leaderLatch = getStartedLeaderLatch(partitionContext);
                if (leaderLatch.hasLeadership()) {
                    return partitionContext;
                }
                //Раз мы не лидер закрываем сессию, если она есть
                partitionContext.closeSession();
                partitionQueue.returnLock(partitionContext);
                Thread.sleep(lockAcquireTimeMillis);
            } catch (Exception e) {
                partitionQueue.returnLock(partitionContext);
                throw e;
            }

        }
    }

    public void releasePartition(PartitionContext context, boolean isError) {
        if (isError) {
            try {
                context.release().get(clickhouseSocketTimeoutSeconds, TimeUnit.SECONDS);
            } catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            } catch (TimeoutException e) {
                log.error("Timeout while waiting for partition to be released. " +
                    "Partition will be released anyway, but it may lead to duplicates.", e);
            }

            releaseLock(context);
        }
        partitionQueue.returnLock(context);
    }

    public List<PartitionContext> getPartitionContexts() {
        return new ArrayList<>(partitionContexts.values());
    }


    private LeaderLatch getStartedLeaderLatch(PartitionContext partitionContext) throws Exception {
        LeaderLatch leaderLatch = partitionContext.getLeaderLatch();
        if (leaderLatch == null) {
            leaderLatch = createLeaderLatch(partitionContext);
        }
        if (leaderLatch.getState() == LeaderLatch.State.LATENT) {
            leaderLatch.start();
            log.info("Starting leader latch for " + partitionContext.getName());
            leaderLatch.await(lockAcquireTimeMillis, TimeUnit.MILLISECONDS);
        }
        return leaderLatch;
    }

    private LeaderLatch createLeaderLatch(PartitionContext partitionContext) throws Exception {
        log.info("Creating leader latch for partition " + partitionContext.getName());
        LeaderLatch leaderLatch = readSemaphore.getDistributedLock(partitionContext.getName(), partitionContext, false);
        partitionContext.setLeaderLatch(leaderLatch);
        return leaderLatch;
    }

    public int getLocksCount() {
        return activeLocks.size();
    }

    private void releaseLock(PartitionContext context) {
        log.info("Releasing lock for partition " + context.getName());
        try {
            context.closeSession();
            if (context.getLeaderLatch() != null) {
                context.getLeaderLatch().close();
                context.setLeaderLatch(null);
            }
            activeLocks.remove(context.getName());
            createLeaderLatch(context);
        } catch (Exception e) {
            log.error("Error while releasing lock for " + context.getName());
        }
    }

    public Multiset<String> getLeaderStat() {
        Multiset<String> stats = HashMultiset.create();
        for (PartitionContext context : partitionContexts.values()) {
            stats.add(getLeader(context));
        }
        return stats;
    }

    private String getLeader(PartitionContext context) {
        LeaderLatch leaderLatch = context.getLeaderLatch();
        if (leaderLatch == null) {
            return "Unknown";
        }
        String leader = null;
        try {
            leader = leaderLatch.getLeader().getId();
        } catch (KeeperException.NoNodeException e) {
            //Нет ноды в ЗК. Значит и лидера нету.
            return "No Leader";
        } catch (Exception e) {
            log.info("Failed to get leader for partition" + context.getName(), e);
            return "Failed to get leader";
        }
        if (leader.isEmpty()) {
            return "No Leader";
        }
        return leader;
    }
}
