package ru.yandex.market.logshatter.reader.logbroker.monitoring;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.SetMultimap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.market.health.KeyValueLog;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.reader.logbroker.LogbrokerSource;
import ru.yandex.market.logshatter.reader.logbroker.MonitoringLagThreshold;
import ru.yandex.market.logshatter.reader.logbroker.PartitionDao;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 17.09.2018
 */
public class LogBrokerLagMonitoring {
    private static final Logger log = LogManager.getLogger();

    private final PartitionDao partitionDao;
    private final List<IdentLagMonitoring> identLagMonitorings;
    private final MonitoringUnit problemWithLagMonitoringMonitoringUnit;

    public LogBrokerLagMonitoring(PartitionDao partitionDao, MonitoringConfig monitoringConfig,
                                  LogShatterMonitoring monitoring, SetMultimap<String, LogbrokerSource> identToSources) {
        this.partitionDao = partitionDao;

        this.problemWithLagMonitoringMonitoringUnit = monitoring.getClusterCritical().createUnit(
            "LogBrokerLagMonitoring",
            monitoringConfig.getReaderMonitoringDelayMinutes(), TimeUnit.MINUTES
        );

        // Крит если мониторинг не обновлялся больше двух минут
        this.problemWithLagMonitoringMonitoringUnit.setCriticalTimeout(2, TimeUnit.MINUTES);

        this.identLagMonitorings = identToSources.keySet().stream()
            .map(ident ->
                new IdentLagMonitoring(
                    ident,
                    identToSources.get(ident),
                    monitoring.getClusterCritical().createUnit(
                        "LogBroker " + ident + " lag",
                        monitoringConfig.getReaderMonitoringDelayMinutes(), TimeUnit.MINUTES
                    ),
                    monitoringConfig.getThreshold(ident)
                )
            )
            .collect(Collectors.toList());
    }

    @Scheduled(cron = "0 * * * * *")  // раз в минуту
    public void run() {
        if (identLagMonitorings.isEmpty()) {
            problemWithLagMonitoringMonitoringUnit.critical("No idents");
            return;
        }
        try {
            ListMultimap<String, LogBrokerOffset> identToOffsets = createIdentToOffsetsMultimap(partitionDao.getAll());
            for (IdentLagMonitoring identLagMonitoring : identLagMonitorings) {
                identLagMonitoring.update(identToOffsets.get(identLagMonitoring.ident));
            }
            problemWithLagMonitoringMonitoringUnit.ok();
        } catch (RuntimeException e) {
            log.error(e);
            problemWithLagMonitoringMonitoringUnit.critical(e);
        }
    }

    private static ListMultimap<String, LogBrokerOffset> createIdentToOffsetsMultimap(List<LogBrokerOffset> offsets) {
        ListMultimap<String, LogBrokerOffset> identToOffsets = ArrayListMultimap.create();
        for (LogBrokerOffset offset : offsets) {
            identToOffsets.put(getIdentByPartition(offset.getPartition()), offset);
        }
        return identToOffsets;
    }

    /**
     * @param partition rt3.sas--market-health-stable--other:34
     * @return market-health-stable
     */
    private static String getIdentByPartition(String partition) {
        int start = partition.indexOf("--");
        int end = partition.lastIndexOf("--");
        if (start != -1 && start + 1 < end) {
            return partition.substring(start + 2, end);
        }
        return null;
    }

    private static class IdentLagMonitoring {
        final String ident;
        final Set<LogbrokerSource> sources;
        final MonitoringUnit monitoringUnit;
        final MonitoringLagThreshold threshold;

        IdentLagMonitoring(String ident, Set<LogbrokerSource> sources, MonitoringUnit monitoringUnit,
                                   MonitoringLagThreshold threshold) {
            this.ident = ident;
            this.sources = sources;
            this.monitoringUnit = monitoringUnit;
            this.threshold = threshold;
        }

        void update(List<LogBrokerOffset> offsets) {
            if (offsets == null || offsets.isEmpty()) {
                monitoringUnit.critical("Unknown for ident: " + ident);
                return;
            }

            IdentLag lag = calculateLag(offsets);

            log.info(
                "LogBroker ident {} lag {} of {} messages ({}%)",
                ident, lag.lagMessages, lag.totalMessages, lag.getLagPercent()
            );

            writeKeyValueLog(lag);

            updateForLag(lag.getLagPercent());
        }

        private IdentLag calculateLag(List<LogBrokerOffset> offsets) {
            IdentLag result = new IdentLag(0 ,0);
            offsets.stream()
                .filter(this::shouldCountOffset)
                .forEach(result::add);
            return result;
        }

        private boolean shouldCountOffset(LogBrokerOffset offset) {
            boolean thereIsConfigForAnyLogType = sources.size() == 1 && sources.iterator().next().getLogType() == null;
            boolean thereAreConfigsForThisLogType = sources.contains(LogbrokerSource.fromPartition(offset.getPartition()));
            return thereIsConfigForAnyLogType || thereAreConfigsForThisLogType;
        }

        private void writeKeyValueLog(IdentLag lag) {
            KeyValueLog.log("lbLagPercent", ident, lag.getLagPercent());
            KeyValueLog.log("lbLag", ident, lag.lagMessages);
            KeyValueLog.log("lbMessages", ident, lag.totalMessages);
        }

        private void updateForLag(double lagPercent) {
            if (lagPercent > threshold.getCriticalPercent()) {
                monitoringUnit.critical(getMonitoringMessage(lagPercent, threshold.getCriticalPercent()));
            } else if (lagPercent > threshold.getWarningPercent()) {
                monitoringUnit.warning(getMonitoringMessage(lagPercent, threshold.getWarningPercent()));
            } else {
                monitoringUnit.ok();
            }
        }

        private String getMonitoringMessage(double lagPercent, double lagThresholdPercent) {
            return String.format(
                "Logbroker %s lag: %.2f%% (more than %.2f%%)",
                ident, lagPercent, lagThresholdPercent
            );
        }
    }

    private static class IdentLag {
        long lagMessages;
        long totalMessages;

        IdentLag(long lagMessages, long totalMessages) {
            this.lagMessages = lagMessages;
            this.totalMessages = totalMessages;
        }

        /**
         * Сообщения всегда нумеруются с 0.
         * https://wiki.yandex-team.ru/logbroker/docs/protocols/legacy-http/read/#offsets.
         *
         * offset.getOffset() - номер последнего обработанного клиентом сообщения, -1 если клиент ещё не читал эту
         * партицию.
         *
         * offset.getLogStart() - номер первого сообщения, которое можно прочитать.
         *
         * offset.getLogEnd() - номер последнего сообщения, которые можно прочитать.
         *
         * offset.getLag() - количество непрочитанных сообщений.
         */
        private void add(LogBrokerOffset offset) {
            lagMessages += offset.getLag();
            totalMessages += offset.getLogEnd() - offset.getLogStart() + 1;
        }

        double getLagPercent() {
            return lagMessages * 100.0 / totalMessages;
        }
    }
}
