package ru.yandex.market.logshatter.reader.logbroker.monitoring;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.market.logbroker.pull.LogBrokerClient;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logbroker.pull.LogBrokerPartition;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.reader.logbroker.LogBrokerConfigurationService;
import ru.yandex.market.logshatter.reader.logbroker.LogbrokerSource;
import ru.yandex.market.logshatter.reader.logbroker.PartitionDao;
import ru.yandex.market.logshatter.reader.logbroker.PartitionManager;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Получает от Логброкера лаг по партициям, которые читает этот инстанс Логшаттера, и записывает его в Монгу.
 * Эти данные потом используют {@link LogBrokerLagMonitoring} и {@link LogBrokerLagLogger}.
 *
 * Ходим в один ДЦ Логброкера (тот, откуда читаем). Он отдаст оффсеты по всем партициям, даже по тем, которые сейчас
 * читаются другими Логшаттерами в других ДЦ. Если партиция сейчас читается в другом ДЦ, то в текущем ДЦ будет
 * неправильный оффсет (устаревшее значение, меньше чем на самом деле). Поэтому для каждой партиции сохраняем в Монгу
 * оффсет только если оффсет в Логброкере больше чем в Монге. За это отвечает метод {@link PartitionDao#advanceOffsets}.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 17.09.2018
 */
public class LogBrokerLagWriterOldApi {
    private static final Logger log = LogManager.getLogger();

    private final LogBrokerClient logBrokerClient;
    private final PartitionDao partitionDao;
    private final PartitionManager partitionManager;
    private final List<LogbrokerSource> sources;
    private final int offsetsFetchingRetryCountUntilCrit;
    private final int offsetsFetchingSleepBetweenRetriesMillis;
    private final MonitoringUnit unableGetOffsetsUnit;

    public LogBrokerLagWriterOldApi(
        LogBrokerClient logBrokerClient,
        PartitionDao partitionDao,
        PartitionManager partitionManager,
        LogBrokerConfigurationService logBrokerConfigurationService,
        LogShatterMonitoring monitoring,
        int offsetsFetchingRetryCountUntilCrit,
        int offsetsFetchingSleepBetweenRetriesMillis
    ) {
        this.logBrokerClient = logBrokerClient;
        this.partitionDao = partitionDao;
        this.partitionManager = partitionManager;
        this.sources = logBrokerConfigurationService.getSources();
        this.offsetsFetchingSleepBetweenRetriesMillis = offsetsFetchingSleepBetweenRetriesMillis;
        this.offsetsFetchingRetryCountUntilCrit = offsetsFetchingRetryCountUntilCrit;

        this.unableGetOffsetsUnit = monitoring.getClusterCritical().createUnit("logbroker: can't get offsets");
        // Крит если не писали оффсеты больше двух минут
        this.unableGetOffsetsUnit.setCriticalTimeout(2, TimeUnit.MINUTES);
    }

    @Scheduled(cron = "0 * * * * *")  // раз в минуту
    public void run() throws InterruptedException {
        List<LogbrokerSource> pendingSources = new ArrayList<>(sources);
        List<LogbrokerSource> failedSources = Collections.emptyList();

        for (int tryNumber = 0; tryNumber < offsetsFetchingRetryCountUntilCrit; ++tryNumber) {
            failedSources = writeLagForSource(pendingSources);

            if (failedSources.isEmpty()) {
                unableGetOffsetsUnit.ok();
                return;
            } else {
                pendingSources = failedSources;
            }

            TimeUnit.MILLISECONDS.sleep(offsetsFetchingSleepBetweenRetriesMillis);
        }

        unableGetOffsetsUnit.critical(
            "Can't get offsets for " + failedSources.size() + " sources " + failedSources
        );
    }


    private List<LogbrokerSource> writeLagForSource(List<LogbrokerSource> pendingSources) {
        List<LogbrokerSource> failedSources = new ArrayList<>();

        for (LogbrokerSource source : pendingSources) {
            try {
                checkPartitions(
                    logBrokerClient.getOffsets(source.getIdent(), source.getLogType()),
                    logBrokerClient.getSuggestPartitions(
                        logBrokerClient.getTopics(source.getIdent(), source.getLogType())
                    )
                );
            } catch (Exception e) {
                log.error("Exception while offsets gathering {}", source, e);
                failedSources.add(source);
            }
        }

        return failedSources;
    }

    private void checkPartitions(List<LogBrokerOffset> offsets, List<LogBrokerPartition> partitions) {
        partitionDao.advanceOffsets(offsets);

        // Всё что ниже перестанет быть нужно после перехода на новое API Логброкера.

        //Сортируем партиции, что бы партиции текущего ДЦ поместить выше и соотвественно приоритетнее их захватить
        partitions.sort((o1, o2) -> {
            if (logBrokerClient.getDc().equals(o1.getPartitionDc())) {
                return -1;
            }
            if (logBrokerClient.getDc().equals(o2.getPartitionDc())) {
                return 1;
            }
            return 0;
        });

        partitionManager.updatePartitions(offsets, partitions);
    }
}
