package ru.yandex.market.logshatter.reader.logbroker.monitoring;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.market.logbroker.pull.LogBrokerSourceKey;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.util.concurrent.TimeUnit;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 24.09.2018
 */
public class LogBrokerSourcesWithoutMetadataMonitoring {
    protected static final Logger log = LogManager.getLogger();

    private final MonitoringConfig monitoringConfig;
    private final MonitoringUnit sourcesWithoutMetadataUnit;

    private final Cache<String, String> sourcesWithoutMetadata = CacheBuilder.newBuilder()
        .expireAfterWrite(1, TimeUnit.HOURS)
        .build();

    public LogBrokerSourcesWithoutMetadataMonitoring(MonitoringConfig monitoringConfig, LogShatterMonitoring monitoring) {
        this.monitoringConfig = monitoringConfig;
        this.sourcesWithoutMetadataUnit = monitoring.getHostCritical().createUnit("logbroker: Sources without metadata");
    }

    public void notifyAboutSourceWithoutMetadata(LogBrokerSourceKey key) {
        if (!monitoringConfig.getSourcesWithoutMetadataWhitelistLogTypes().contains(key.getType())) {
            if (sourcesWithoutMetadata.getIfPresent(key.getSourceId()) == null) {
                log.error("No metadata for lb source key: {}", key);
            }
            sourcesWithoutMetadata.put(key.getSourceId(), key.getSourceId());
        }
    }

    @Scheduled(cron = "0 * * * * *")  // раз в минуту
    public void run() {
        sourcesWithoutMetadata.cleanUp();
        long count = sourcesWithoutMetadata.size();
        if (count >= monitoringConfig.getSourcesWithoutMetadataForCrit()) {
            sourcesWithoutMetadataUnit.critical(
                "Too many sources (" + count + ") without metadata " +
                    "(>=" + monitoringConfig.getSourcesWithoutMetadataForCrit() + ")."
            );
        } else if (count > 0) {
            sourcesWithoutMetadataUnit.warning("Found " + count + " source(s) without metadata.");
        } else {
            sourcesWithoutMetadataUnit.ok();
        }
    }
}
