package ru.yandex.market.logshatter.reader.logbroker2;

import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.AbstractScheduledService;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.logshatter.reader.logbroker2.dc.LbDataCenterReaderService;
import ru.yandex.market.logshatter.reader.logbroker2.dc.LbDataCenterReaderServiceFactory;
import ru.yandex.market.logshatter.reader.logbroker2.threads.SingleThreadExecutorServiceFactory;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Логшаттер должен уметь переживать отключение одного любого ДЦ. Это означает что нам не подходит обычная схема
 * логброкерных консьюмеров "консьюмер живёт в одном ДЦ и читает оттуда все данные". Логшаттер умеет жить в нескольких
 * ДЦ.
 *
 * Топики и партиции не привязаны жёстко к какому-то инстансу или ДЦ Логшаттера. Каждый инстанс Логшаттера умеет читать
 * из всех ДЦ Логброкера и имеет доступ ко всем данным.
 *
 * В идеале нужно чтобы каждый инстанс Логшаттера в первую очередь читал данные из своего ДЦ, чтобы не было ситуаций
 * когда Логшаттер в Сасово читает данные из Логброкера в Ивантеевке, а Логшаттер в Ивантеевке читает данные из
 * Логброкера в Сасово. Но это пока не реализовано. st/MARKETINFRA-4367.
 *
 * Этот класс создаёт по {@link LbDataCenterReaderService} на каждый ДЦ, из которого пишутся данные и рестартит
 * {@link LbDataCenterReaderService}'ы если они падают или завершаются самопроизвольно.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 26.12.2018
 */
public class LogBrokerReaderService2 extends AbstractScheduledService {
    private static final Logger log = LogManager.getLogger();

    private final SingleThreadExecutorServiceFactory singleThreadExecutorServiceFactory;
    private final LbDataCenterReaderServiceFactory dataCenterReaderServiceFactory;
    private final ImmutableList<String> dataCentersWithLogBroker;

    private final Map<String, LbDataCenterReaderService> dataCenterToReaderService = new HashMap<>();

    public LogBrokerReaderService2(
        SingleThreadExecutorServiceFactory singleThreadExecutorServiceFactory,
        LbDataCenterReaderServiceFactory dataCenterReaderServiceFactory,
        Collection<String> dataCentersWithLogBroker
    ) {
        this.singleThreadExecutorServiceFactory = singleThreadExecutorServiceFactory;
        this.dataCenterReaderServiceFactory = dataCenterReaderServiceFactory;
        this.dataCentersWithLogBroker = ImmutableList.copyOf(dataCentersWithLogBroker);
    }

    @Override
    protected ScheduledExecutorService executor() {
        return singleThreadExecutorServiceFactory.create(getClass().getSimpleName());
    }

    @Override
    protected Scheduler scheduler() {
        return Scheduler.newFixedRateSchedule(0, 1, TimeUnit.MINUTES);
    }

    @Override
    protected void startUp() {
        log.info("Starting LogBrokerReaderService");
    }

    @Override
    protected void shutDown() {
        log.info("Stopping LogBrokerReaderService");
    }

    @Override
    protected void runOneIteration() {
        // TODO MARKETINFRA-1826 отключать чтение из тех ДЦ, где уже есть работающий Логшаттер
        dataCentersWithLogBroker.forEach(this::ensureThatReaderServiceForDataCenterIsWorking);
    }

    private void ensureThatReaderServiceForDataCenterIsWorking(String dataCenter) {
        if (shouldStartDataCenterReaderService(dataCenter)) {
            startDataCenterReaderService(dataCenter);
        }
    }

    private boolean shouldStartDataCenterReaderService(String dataCenter) {
        LbDataCenterReaderService existingTopicReaderService = dataCenterToReaderService.get(dataCenter);
        return existingTopicReaderService == null
            || existingTopicReaderService.state() == State.TERMINATED
            || existingTopicReaderService.state() == State.FAILED;
    }

    private void startDataCenterReaderService(String dataCenter) {
        LbDataCenterReaderService newService = dataCenterReaderServiceFactory.create(dataCenter);
        dataCenterToReaderService.put(dataCenter, newService);
        newService.startAsync();
    }
}
