package ru.yandex.market.logshatter.reader.logbroker2.common;

import com.google.common.base.Preconditions;

import java.util.regex.Pattern;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 07.12.2018
 */
public class TopicId {
    private static final Pattern TOPIC_ID_PARTS_SPLIT_REGEX = Pattern.compile("--");
    private static final Pattern DATA_CENTER_PARTS_SPLIT_REGEX = Pattern.compile("\\.");
    private final String asString;

    private final String dataCenter;
    private final String ident;
    private final String logType;

    private TopicId(String asString, String dataCenter, String ident, String logType) {
        this.asString = asString;
        this.dataCenter = dataCenter;
        this.ident = ident;
        this.logType = logType;
    }

    /**
     * @param topic Примеры: "rt3.man--market-health-testing--other", "market-health-testing--other"
     */
    public static TopicId fromString(String topic) {
        String[] split = TOPIC_ID_PARTS_SPLIT_REGEX.split(topic);

        Preconditions.checkArgument(
            split.length >= 2 && split.length <= 3,
            String.format("There are %d parts in topic '%s', expected 2 or 3", split.length, topic)
        );

        if (split.length == 3) {
            String[] splitDataCenter = DATA_CENTER_PARTS_SPLIT_REGEX.split(split[0]);

            Preconditions.checkArgument(
                splitDataCenter.length == 2,
                String.format("There are %d parts in data center part of topic '%s', expected 2", split.length, topic)
            );

            return new TopicId(
                topic,
                splitDataCenter[1],
                split[1],
                split[2]
            );
        }

        return new TopicId(
            topic,
            null,
            split[0],
            split[1]
        );
    }

    public String asString() {
        return asString;
    }

    public String asStringWithoutDataCenter() {
        return ident + "--" + logType;
    }

    public String getDataCenter() {
        return dataCenter;
    }

    public String getIdent() {
        return ident;
    }

    public String getLogType() {
        return logType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        return asString.equals(((TopicId) o).asString);
    }

    @Override
    public int hashCode() {
        return asString.hashCode();
    }
}
