package ru.yandex.market.logshatter.reader.logbroker2.dc;

import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.AbstractScheduledService;
import com.google.common.util.concurrent.Service;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.health.KeyValueLog;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.reader.logbroker.LogbrokerSource;
import ru.yandex.market.logshatter.reader.logbroker.PartitionDao;
import ru.yandex.market.logshatter.reader.logbroker2.common.TopicId;
import ru.yandex.market.logshatter.reader.logbroker2.threads.SingleThreadExecutorServiceFactory;
import ru.yandex.market.logshatter.reader.logbroker2.topic.LbTopicReaderService;
import ru.yandex.market.logshatter.reader.logbroker2.topic.LbTopicReaderServiceFactory;
import ru.yandex.market.monitoring.MonitoringUnit;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Этот класс создаёт по {@link LbTopicReaderService} на каждый топик, упомянутый в конфигах Логшаттера и рестартит
 * {@link LbTopicReaderService}'ы если они падают или завершаются самопроизвольно.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 26.12.2018
 */
public class LbDataCenterReaderService extends AbstractScheduledService {
    private static final DateTimeFormatter DATE_TIME_FORMATTER =
        DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSXXX").withZone(ZoneId.systemDefault());

    private static final Logger log = LogManager.getLogger();
    private static final Logger lbSessionsLog = LogManager.getLogger("lbSessions");

    private final String dataCenter;
    private final LbTopicReaderServiceFactory topicReaderServiceFactory;
    private final SingleThreadExecutorServiceFactory singleThreadExecutorServiceFactory;

    private final LbOldApiOffsetsFetcher offsetsFetcher;

    private final List<LogbrokerSource> sources;
    private final PartitionDao partitionDao;
    private final MonitoringUnit monitoringUnitRefresh;
    private final MonitoringUnit monitoringUnitOffsets;

    private final Map<TopicId, LbTopicReaderService> topicIdToTopicReaderServiceMap = new HashMap<>();

    LbDataCenterReaderService(
        String dataCenter,
        LbTopicReaderServiceFactory topicReaderServiceFactory,
        SingleThreadExecutorServiceFactory singleThreadExecutorServiceFactory,
        LbOldApiOffsetsFetcher offsetsFetcher, List<LogbrokerSource> sources,
        PartitionDao partitionDao,
        LogShatterMonitoring monitoring
    ) {
        this.dataCenter = dataCenter;
        this.topicReaderServiceFactory = topicReaderServiceFactory;
        this.singleThreadExecutorServiceFactory = singleThreadExecutorServiceFactory;
        this.offsetsFetcher = offsetsFetcher;
        this.sources = ImmutableList.copyOf(sources);
        this.partitionDao = partitionDao;

        this.monitoringUnitRefresh = monitoring.getClusterCritical()
            .createUnit("failed to refresh DC " + dataCenter);
        this.monitoringUnitRefresh.setCriticalTimeout(2, TimeUnit.MINUTES);

        this.monitoringUnitOffsets = monitoring.getClusterCritical()
            .createUnit("failed to fetch offsets from DC " + dataCenter);
        this.monitoringUnitOffsets.setCriticalTimeout(2, TimeUnit.MINUTES);
    }

    @Override
    protected String serviceName() {
        return "dc_" + dataCenter;
    }

    @Override
    protected ScheduledExecutorService executor() {
        return singleThreadExecutorServiceFactory.create(serviceName());
    }

    @Override
    protected Scheduler scheduler() {
        return Scheduler.newFixedRateSchedule(0, 1, TimeUnit.MINUTES);
    }

    @Override
    protected void startUp() {
        log.info("Starting reading from dataCenter={}", dataCenter);
    }

    @Override
    protected void shutDown() {
        log.info("Stopping reading from dataCenter={}", dataCenter);
        topicIdToTopicReaderServiceMap.values().forEach(Service::stopAsync);
    }

    @Override
    protected void runOneIteration() {
        try {
            // Получаем оффсеты для всех первородных партиций всех идентов и топиков, которые указаны в конфигах
            // Логшаттера. Если для каких-то идентов и топиков не удалось получить оффсеты, то не падаем, а зажигаем
            // мониторинг и создаём сессии для тех тех топиков, для которых удалось получить оффсеты.
            LbOldApiOffsetsFetcher.OffsetsFetchingResult offsets = fetchOffsets();

            // Создаём сессии для топиков, для которых ещё нет сессий или сессии которых упали. Не пытаемся закрыть
            // сессии топиков, которых нет в списке. Считаем что если топик действительно удалили или забрали на него
            // права, то они сами упадут и порестарчены не будут. Плюс не хотим закрывать сессии топиков, для которых
            // временно не смогли получить оффсеты.
            ensureThatTopicSessionsAreOpen(offsets);

            // Сохраняем оффсеты в Монгу чтобы работал график и мониторинг лага.
            // См LogBrokerLagLogger и LogBrokerLagMonitoring. Прямо здесь лаг посчитать не можем, потому что здесь
            // информация только за один ДЦ.
            saveOffsetsToMongo(offsets);

            logSessionStates();

            logLogBatchQueueStates();

            monitoringUnitRefresh.ok();
        } catch (Exception e) {
            log.error("Failed to refresh LbDataCenterReaderService for dataCenter=" + dataCenter, e);
            monitoringUnitRefresh.critical("dc=" + dataCenter, e);
        }
    }

    private LbOldApiOffsetsFetcher.OffsetsFetchingResult fetchOffsets() {
        log.info("Fetching offsets from LogBroker for dataCenter={}", dataCenter);

        Stopwatch stopwatch = Stopwatch.createStarted();

        LbOldApiOffsetsFetcher.OffsetsFetchingResult offsets = offsetsFetcher.fetchOffsets(sources);
        updateOffsetsFetchingMonitoring(offsets);

        KeyValueLog.log("fetchOffsets", dataCenter, stopwatch.elapsed().toMillis());
        KeyValueLog.log("fetchOffsetsErrorCount", dataCenter, offsets.getFailedSources().size());

        return offsets;
    }

    private void updateOffsetsFetchingMonitoring(LbOldApiOffsetsFetcher.OffsetsFetchingResult offsets) {
        if (offsets.getFailedSources().isEmpty()) {
            monitoringUnitOffsets.ok();
        } else {
            monitoringUnitOffsets.critical(
                offsets.getFailedSources().stream()
                    .map(Object::toString)
                    .collect(Collectors.joining(", "))
            );
        }
    }

    private void ensureThatTopicSessionsAreOpen(LbOldApiOffsetsFetcher.OffsetsFetchingResult offsets) {
        log.info("Creating read sessions for dataCenter={}", dataCenter);

        Stopwatch stopwatch = Stopwatch.createStarted();

        offsets.getTopicIds().stream()
            .filter(topicId -> Objects.equals(topicId.getDataCenter(), dataCenter))
            .forEach(this::ensureThatTopicSessionIsOpen);

        KeyValueLog.log("ensureThatTopicSessionsAreOpen", dataCenter, stopwatch.elapsed().toMillis());
    }

    private void ensureThatTopicSessionIsOpen(TopicId topicId) {
        if (shouldStartSessionForTopicId(topicId)) {
            startSessionForTopicId(topicId);
        }
    }

    private boolean shouldStartSessionForTopicId(TopicId topicId) {
        LbTopicReaderService existingTopicReaderService = topicIdToTopicReaderServiceMap.get(topicId);
        return existingTopicReaderService == null
            || existingTopicReaderService.state() == State.TERMINATED
            || existingTopicReaderService.state() == State.FAILED;
    }

    private void startSessionForTopicId(TopicId topicId) {
        LbTopicReaderService newService = topicReaderServiceFactory.create(topicId);
        topicIdToTopicReaderServiceMap.put(topicId, newService);
        newService.startAsync();
    }

    private void saveOffsetsToMongo(LbOldApiOffsetsFetcher.OffsetsFetchingResult offsets) {
        log.info("Saving offsets to Mongo for dataCenter={}", dataCenter);

        Stopwatch stopwatch = Stopwatch.createStarted();

        partitionDao.advanceOffsets(offsets.getOffsets());

        KeyValueLog.log("saveOffsetsToMongo", dataCenter, stopwatch.elapsed().toMillis());
    }

    private void logSessionStates() {
        for (LbTopicReaderService topicReaderService : topicIdToTopicReaderServiceMap.values()) {
            lbSessionsLog.info(new TskvRecordBuilder()
                .add("date", DATE_TIME_FORMATTER.format(Instant.now()))
                .add("dc", dataCenter)
                .add("ident", topicReaderService.getTopicId().getIdent())
                .add("topic", topicReaderService.getTopicId().asString())
                .add("state", topicReaderService.state())
                .add("compression_ratio", topicReaderService.getCompressionRatio().orElse(-1.0))
                .build()
            );
        }
    }

    private void logLogBatchQueueStates() {
        for (LbTopicReaderService topicReaderService : topicIdToTopicReaderServiceMap.values()) {
            topicReaderService.logLogBatchQueueStates();
        }
    }
}
