package ru.yandex.market.logshatter.reader.logbroker2.dc;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.common.util.concurrent.ThreadFactories;
import ru.yandex.market.logbroker.pull.LogBrokerClient;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logshatter.reader.logbroker.LogbrokerSource;
import ru.yandex.market.logshatter.reader.logbroker2.common.TopicId;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 01.03.2019
 */
class LbOldApiOffsetsFetcher {
    private static final Logger log = LogManager.getLogger();

    private final String dataCenter;
    private final LogBrokerClient logBrokerClient;

    private final int offsetsFetchingRetryCountUntilCrit;
    private final int offsetsFetchingSleepBetweenRetriesMillis;

    private final ExecutorService executorService;

    LbOldApiOffsetsFetcher(
        String dataCenter,
        LogBrokerClient logBrokerClient,
        int offsetsFetchingRetryCountUntilCrit,
        int offsetsFetchingSleepBetweenRetriesMillis
    ) {
        this.dataCenter = dataCenter;
        this.logBrokerClient = logBrokerClient;
        this.offsetsFetchingRetryCountUntilCrit = offsetsFetchingRetryCountUntilCrit;
        this.offsetsFetchingSleepBetweenRetriesMillis = offsetsFetchingSleepBetweenRetriesMillis;
        this.executorService = Executors.newFixedThreadPool(
            5,
            ThreadFactories.named(getClass().getSimpleName() + "_" + dataCenter)
        );
    }

    OffsetsFetchingResult fetchOffsets(List<LogbrokerSource> sources) {
        OffsetsFetchingResult result = new OffsetsFetchingResult();

        List<Future<?>> futures = sources.stream()
            .map(source -> executorService.submit(() -> result.fetchOffsetsForSource(source)))
            .collect(Collectors.toList());

        for (Future<?> future : futures) {
            try {
                future.get();
            } catch (InterruptedException | ExecutionException e) {
                // Сюда попадать не должны, ошибки обрабатываются в fetchOffsetsForSource
                throw new RuntimeException(e);
            }
        }

        return result;
    }

    public class OffsetsFetchingResult {
        private final List<LogBrokerOffset> offsets = Collections.synchronizedList(new ArrayList<>());
        private final List<LogbrokerSource> failedSources = Collections.synchronizedList(new ArrayList<>());

        void fetchOffsetsForSource(LogbrokerSource source) {
            for (int i = 0; i < offsetsFetchingRetryCountUntilCrit - 1; i++) {
                try {
                    offsets.addAll(fetchOffsetsForSourceWithoutRetries(source));
                } catch (Exception e) {
                    log.error("Exception while offsets gathering {}", source, e);
                    try {
                        TimeUnit.MILLISECONDS.sleep(offsetsFetchingSleepBetweenRetriesMillis);
                    } catch (InterruptedException e1) {
                        throw new RuntimeException(e1);
                    }
                }
            }
            try {
                offsets.addAll(fetchOffsetsForSourceWithoutRetries(source));
            } catch (Exception e) {
                log.error("Exception while offsets gathering {}", source, e);
                failedSources.add(source);
            }
        }

        private List<LogBrokerOffset> fetchOffsetsForSourceWithoutRetries(LogbrokerSource source) throws Exception {
            List<LogBrokerOffset> fetchedOffsets =
                logBrokerClient.getOffsets(source.getIdent(), source.getLogType()).stream()
                    .filter(offset -> Objects.equals(dataCenter, offset.getDc()))
                    .collect(Collectors.toList());
            if (fetchedOffsets.isEmpty()) {
                throw new RuntimeException("LogBroker returned empty offsets list for source " + source);
            }
            return fetchedOffsets;
        }

        public List<LogBrokerOffset> getOffsets() {
            return offsets;
        }

        public List<TopicId> getTopicIds() {
            return offsets.stream()
                .map(offset -> offset.getPartition().split(":")[0])
                .distinct()
                .map(TopicId::fromString)
                .collect(Collectors.toList());
        }

        public List<LogbrokerSource> getFailedSources() {
            return failedSources;
        }
    }
}
