package ru.yandex.market.logshatter.reader.logbroker2.topic;

import com.google.common.annotations.VisibleForTesting;

import javax.annotation.concurrent.ThreadSafe;

/**
 * st/MARKETINFRA-4396.
 *
 * У Логшаттера в памяти есть очереди данных, ожидающих парсинга и записи в Кликхаус. Эти очереди должны быть ограничены
 * про размеру чтобы не было OOM когда парсинг или сохранение в Кликхаус тормозят.
 *
 * В gRPC API Логброкера нет поля с размером сжатых данных. Приходят только сами сжатые данные. Расжатием занимаются
 * потоки-парсеры. Нельзя разжимать данные сразу при чтении, потому что там один поток на сессию и одна сессия на топик.
 * Считаем что в общем случае один поток не справится с расжатием целого топика.
 *
 * Получается что нужно добавить сжатые данные в очередь на парсинг не зная размер разжатых данных. Это проблема, потому
 * что при добавлении данных в очередь нужно поинкрементить счётчик размера данных в очереди.
 *
 * Этот класс вычисляет коэффициент сжатия топика чтобы можно было вычислить приблизительный размер разжатых данных не
 * разжимая их.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 14.03.2019
 */
@ThreadSafe
public class CompressionRatioCalculator {
    @VisibleForTesting
    public static final int MIN_COMPRESSED_SIZE_FOR_PRECISE_COMPRESSION_RATIO_CALCULATION_BYTES = 32 * 1024 * 1024;
    public static final int DEFAULT_COMPRESSION_RATIO = 20;

    private long totalCompressedSizeBytes = 0;
    private long totalDecompressedSizeBytes = 0;

    synchronized void add(long compressedSizeBytes, long decompressedSizeBytes) {
        totalCompressedSizeBytes += compressedSizeBytes;
        totalDecompressedSizeBytes += decompressedSizeBytes;
    }

    synchronized boolean canCalculatePreciseCompressionRatio() {
        // Проверяем что уже насобирали достаточно статистики и что значение totalDecompressedSizeBytes адекватно
        return (totalCompressedSizeBytes > MIN_COMPRESSED_SIZE_FOR_PRECISE_COMPRESSION_RATIO_CALCULATION_BYTES)
            && (totalDecompressedSizeBytes >= totalCompressedSizeBytes);
    }

    synchronized double getCompressionRatio() {
        if (!canCalculatePreciseCompressionRatio()) {
            // Если не можем посчитать по-честному, возвращаем безопасное завышенное значение
            return DEFAULT_COMPRESSION_RATIO;
        }
        return (double) totalDecompressedSizeBytes / totalCompressedSizeBytes;
    }
}
