package ru.yandex.market.logshatter.reader.logbroker2.topic;

import com.google.common.base.Stopwatch;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 11.10.2018
 */
public class LbApiActionLogger {
    private static final Logger log = LogManager.getLogger();

    private static final DateTimeFormatter DATE_TIME_FORMATTER =
        DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSXXX").withZone(ZoneId.systemDefault());

    private final Consumer<String> logger;
    private final String actionName;
    private final Stopwatch stopwatch;
    private final Function<TskvRecordBuilder, TskvRecordBuilder> fieldsThatShouldBeAddedToEveryStage;

    public LbApiActionLogger(Consumer<String> logger, String actionName,
                             Function<TskvRecordBuilder, TskvRecordBuilder> fieldsThatShouldBeAddedToEveryStage) {
        this.logger = logger;
        this.actionName = actionName;
        this.fieldsThatShouldBeAddedToEveryStage = fieldsThatShouldBeAddedToEveryStage;
        this.stopwatch = Stopwatch.createStarted();
    }


    public LbApiActionLogger start() {
        return start(builder -> builder);
    }

    public LbApiActionLogger start(Function<TskvRecordBuilder, TskvRecordBuilder> startFields) {
        return stage("start", startFields);
    }


    public LbApiActionLogger success() {
        return success(builder -> builder);
    }

    public LbApiActionLogger success(Function<TskvRecordBuilder, TskvRecordBuilder> successFields) {
        return timedStage("success", successFields);
    }


    public LbApiActionLogger failure(Throwable throwable) {
        return failure(throwable, builder -> builder);
    }

    public LbApiActionLogger failure(Throwable throwable, Function<TskvRecordBuilder, TskvRecordBuilder> failureFields) {
        log.error("", throwable);
        return timedStage(
            "failure",
            failureFields.andThen(builder -> builder.add("exception", throwable))
        );
    }


    public LbApiActionLogger timedStage(String stageName) {
        return timedStage(stageName, builder -> builder);
    }

    public LbApiActionLogger timedStage(String stageName, Function<TskvRecordBuilder, TskvRecordBuilder> fields) {
        return stage(
            stageName,
            fields.andThen(builder -> builder.add("time_since_start_ms", stopwatch.elapsed(TimeUnit.MILLISECONDS)))
        );
    }


    public LbApiActionLogger stage(String stageName) {
        return stage(stageName, builder -> builder);
    }

    public LbApiActionLogger stage(String stageName, Function<TskvRecordBuilder, TskvRecordBuilder> fields) {
        logger.accept(
            fields.apply(
                fieldsThatShouldBeAddedToEveryStage.apply(
                    new TskvRecordBuilder()
                        .add("date", DATE_TIME_FORMATTER.format(Instant.now()))
                        .add("action", actionName)
                        .add("stage", stageName)
                )
            )
                .build()
        );
        return this;
    }
}
