package ru.yandex.market.logshatter.reader.logbroker2.topic;

import io.grpc.StatusRuntimeException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerLockMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReleaseMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.market.logshatter.reader.logbroker2.common.TopicId;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.util.Collection;
import java.util.Objects;
import java.util.function.Function;

/**
 * Этот класс занимается логированием сообщений, приходящих от Логброкера, и реакции Логшаттера на них. Методы этого
 * класса записывают одну или несколько строк в лог и возвращают {@link LbApiActionLogger}, в который можно записать больше
 * строк.
 *
 * LbApiLogger и ActionLogger лучше чем просто голые Logger и TskvRecordBuilder, потому что они умеют автоматически
 * дописывать общие поля ко всем строкам лога, относящимся к обработке одного сообщения от Логброкера (дата и время,
 * sessionId и так далее).
 *
 * А ещё ActionLogger умеет измерять время, затраченное на обработку сообщения от Логброкера.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 13.09.2018
 */
class LbApiLogger {
    private static final Logger log = LogManager.getLogger("lbApi");

    private final String sessionId;
    private final TopicId topicId;

    LbApiLogger(String sessionId, TopicId topicId) {
        this.sessionId = sessionId;
        this.topicId = topicId;
    }

    LbApiActionLogger init() {
        return createActionLogger("init").start();
    }

    LbApiActionLogger read(ConsumerReadResponse read) {
        LbApiActionLogger actionLogger = createActionLogger("read");

        // Здесь нельзя вызывать MessageData.getDecompressedData, потому что логброкерные сессии однопоточные, и не
        // справятся с такой нагрузкой. Распаковка данных происходит в потоках, которые парсят.
        if (read.getBatches() != null) {
            actionLogger.start(
                builder -> builder
                    // В одном ConsumerReadResponse'е может быть несколько партиций.
                    // В большинстве случаев одна. Записываем первую.
                    .add("partition", read.getBatches().isEmpty() ? -1 : read.getBatches().get(0).getPartition())
                    .add("cookie", read.getCookie())
                    .add("batch_count", read.getBatches().size())
                    .add(
                        "messages_count",
                        read.getBatches().stream()
                            .map(MessageBatch::getMessageData)
                            .filter(Objects::nonNull)
                            .mapToLong(Collection::size)
                            .sum()
                    )
                    .add(
                        "compressed_data_size",
                        read.getBatches().stream()
                            .map(MessageBatch::getMessageData)
                            .filter(Objects::nonNull)
                            .flatMap(Collection::stream)
                            .filter(Objects::nonNull)
                            .map(MessageData::getRawData)
                            .filter(Objects::nonNull)
                            .mapToLong(rawData -> rawData.length)
                            .sum()
                    )
            );
        }

        return actionLogger;
    }

    LbApiActionLogger lock(ConsumerLockMessage lock) {
        return createActionLogger(
            "lock",
            builder -> builder
                .add("partition", lock.getPartition())
        )
            .start(
                builder -> builder
                    .add("read_offset", lock.getReadOffset())
                    .add("end_offset", lock.getEndOffset())
                    .add("generation", lock.getGeneration())
            );
    }

    LbApiActionLogger release(ConsumerReleaseMessage release) {
        return createActionLogger(
            "release",
            builder -> builder
                .add("partition", release.getPartition())
        )
            .start(
                builder -> builder
                    .add("can_commit", release.isCanCommit())
                    .add("generation", release.getGeneration())
            );
    }

    LbApiActionLogger close() {
        return createActionLogger("close").start();
    }

    LbApiActionLogger error(Throwable throwable) {
        return createActionLogger("error")
            .start(
                builder -> builder
                    .add("error", throwable.toString())
                    .add("low_cardinality_error", convertThrowableToLowCardinalityErrorString(throwable))
            );
    }

    /**
     * low_cardinality_error - поле в КХ, по которому строится график падения сессий с разделением по причине. Туда
     * нельзя записывать всякие id сессий и прочие уникальные вещи. Делаем здесь whitelist исключений, для которых можно
     * сохранять message, для остальных записываем только класс.
     */
    private static String convertThrowableToLowCardinalityErrorString(Throwable throwable) {
        if (throwable instanceof StatusRuntimeException) {
            return throwable.toString();
        }
        return throwable.getClass().getName();
    }

    private LbApiActionLogger createActionLogger(String actionName) {
        return createActionLogger(actionName, builder -> builder);
    }

    private LbApiActionLogger createActionLogger(
        String actionName, Function<TskvRecordBuilder, TskvRecordBuilder> fieldsThatShouldBeAddedToEveryStage
    ) {
        return new LbApiActionLogger(
            log::info,
            actionName,
            builder -> fieldsThatShouldBeAddedToEveryStage
                .apply(builder
                    .add("session_id", sessionId)
                    .add("topic_id", topicId.asString())
                )
        );
    }
}
