package ru.yandex.market.logshatter.reader.logbroker2.topic;

import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.consumer.StreamConsumer;
import ru.yandex.kikimr.persqueue.consumer.stream.StreamConsumerConfig;
import ru.yandex.market.logshatter.reader.logbroker2.common.TopicId;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 12.10.2018
 */
public class LbApiStreamConsumerFactory {
    private final Map<String, LogbrokerClientFactory> dataCenterToLogBrokerClientFactoryMap;
    private final LbCredentialsProvider credentialsProvider;
    private final String clientId;
    private final int maxReadBatchSize;
    private final int maxInflightReads;
    private final int maxUnconsumedReads;

    public LbApiStreamConsumerFactory(
        Map<String, LogbrokerClientFactory> dataCenterToLogBrokerClientFactoryMap,
        LbCredentialsProvider credentialsProvider,
        String clientId,
        int maxReadBatchSize,
        int maxInflightReads,
        int maxUnconsumedReads
    ) {
        this.dataCenterToLogBrokerClientFactoryMap = dataCenterToLogBrokerClientFactoryMap;
        this.clientId = clientId;
        this.credentialsProvider = credentialsProvider;
        this.maxReadBatchSize = maxReadBatchSize;
        this.maxInflightReads = maxInflightReads;
        this.maxUnconsumedReads = maxUnconsumedReads;
    }

    /**
     * @param executorService Экзекьютор, который выполняет задачи в отдельном потоке и гарантирует что задачи будут
     *                        выполняться в том порядке, в котором они были добавлены. По умолчанию
     *                        LogbrokerClientFactory использует MoreExecutors.newDirectExecutorService. Он не подходит,
     *                        потому что если его использовать, то сообщения от Логброкера будут обрабатываться
     *                        GRPC-шным пулом потоков. Так делать нельзя, потому что при обработке данных от Логброкера
     *                        может понадобиться долго ждать освобождения места в очереди на парсинг и запись в
     *                        Кликхаус.
     */
    public StreamConsumer createStreamConsumer(TopicId topic, ExecutorService executorService) {
        try {
            return Optional.ofNullable(dataCenterToLogBrokerClientFactoryMap.get(topic.getDataCenter()))
                .orElseThrow(() -> new RuntimeException(String.format(
                    "There is no LogBroker in data center '%s'",
                    topic.getDataCenter()
                )))
                .streamConsumer(
                    StreamConsumerConfig.builder(Collections.singletonList(topic.asStringWithoutDataCenter()), clientId)
                        .setExecutor(executorService)
                        .setCredentialsProvider(credentialsProvider)
                        .configureSession(builder -> builder
                            // Когда readOnlyLocal=true, Логброкер будет присылать только данные, которые были записаны
                            // в том ДЦ, из которого читаем. Когда readOnlyLocal=false, Логброкер будет присылать все
                            // данные из всех ДЦ. Мы ходим во все ДЦ Логброкера, поэтому из каждого ДЦ можно читать
                            // только те данные, которые были записаны в этом ДЦ.
                            .setReadOnlyLocal(true)
                            // Когда clientSideLocksAllowed=true, Логброкер будет присылать сообщения Lock для каждой
                            // партиции и дожидаться Locked в ответ прежде чем присылать данные из этой партиции.
                            // Это единственный правильный способ для Логшаттера узнать список партиций. Ещё в таком
                            // режиме Логброкер будет балансировать партиции между Логшаттерами, которые читают из
                            // одного ДЦ. Подробнее здесь: https://nda.ya.ru/3UXih6
                            .setClientSideLocksAllowed(true)
                            // Когда forceBalancePartitions=false, при перебалансировке партиций между Логшаттерами
                            // Логброкер будет дожидаться коммита всех оффсетов прежде чем отдать партицию другому
                            // Логшаттеру. Это гарантирует что ни в какой момент времени два Логшаттера не будут читать
                            // одну партицию. Подробнее в LbTopicReaderService.StreamListenerImpl#onLock и
                            // LbTopicReaderService.StreamListenerImpl#onRelease
                            .setForceBalancePartitions(false)
                            // Если ничего не прочитали за 10 минут, то сессия порестартится
                            .setIdleTimeoutSec(10 * 60)
                        )
                        .configureReader(builder -> builder
                            // Максимальное количество сообщений в пачке данных. Нет смысла ограничивать, потому что
                            // лучше ограничить максимальный размер пачки в байтах.
                            .setMaxCount(Integer.MAX_VALUE)
                            // Максимальный размер пачки сообщений от Логброкера в байтах. Маленькие значения создадут
                            // оверхед при запросе/получении пачек данных (много маленьких запросов). Большие значения
                            // приведут к тому, что если парсинг или сохранение в Кликхаус идут медленно, то в памяти
                            // будут лежать maxUnconsumedReads больших пачек с данными.
                            .setMaxSize(maxReadBatchSize)
                            // Количество параллельных запросов к Логброкеру, которые просят новую пачку данных.
                            .setMaxInflightReads(maxInflightReads)
                            // Количество пачек данных, которые могут стоять в очереди на обработку.
                            .setMaxUnconsumedReads(maxUnconsumedReads)
                        )
                        .configureCommiter(builder -> builder
                            .setMaxUncommittedReads(Integer.MAX_VALUE)
                        )
                        .build()
                );
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
