package ru.yandex.market.logshatter.reader.logbroker2.topic;

import com.google.common.base.Strings;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.CountingInputStream;
import com.google.common.util.concurrent.AbstractService;
import org.anarres.lzo.LzopInputStream;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;
import ru.yandex.kikimr.persqueue.compression.UnsupportedCodecException;
import ru.yandex.kikimr.persqueue.consumer.StreamConsumer;
import ru.yandex.kikimr.persqueue.consumer.StreamListener;
import ru.yandex.kikimr.persqueue.consumer.stream.BasicStreamListener;
import ru.yandex.kikimr.persqueue.consumer.transport.message.CommitMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerInitResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerLockMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReleaseMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logbroker.pull.LogBrokerSourceKey;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.LogShatterService;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.SourceContext;
import ru.yandex.market.logshatter.reader.logbroker.LogBrokerPartitionSourceContextsStorage;
import ru.yandex.market.logshatter.reader.logbroker.LogbrokerSourceContext;
import ru.yandex.market.logshatter.reader.logbroker.PartitionDao;
import ru.yandex.market.logshatter.reader.logbroker2.common.PartitionIdUtils;
import ru.yandex.market.logshatter.reader.logbroker2.common.TopicId;
import ru.yandex.market.logshatter.reader.logbroker2.threads.SingleThreadExecutorServiceFactory;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;

/**
 * Этот класс занимается обработкой сообщений от Логброкера, относящихся к одной сессии чтения. Для чтения каждого
 * топика создаётся отдельная сессия чтения.
 *
 * Расшифровка {@link #state}:
 * - NEW - Ещё пока ничего не произошло
 * - STARTING - создаём поток, коннектимся к Логброкеру, ждём сообщение Init от Логброкера
 * - RUNNING - получили Init от Логброкера, реагируем на сообщения от Логброкера
 * - STOPPING - закрываем соединение с Логброкером, останавливаем поток
 * - TERMINATED - соединение с Логброкером закрыто, поток остановлен
 * - FAILED - Логброкер прислал ошибку или при обработке какого-то сообщения от Логброкера вылетело исключение
 *
 * Этот класс не поддерживает рестарт. Если он в FAILED, то надо чтобы кто-то выше создал и запустил новый экземпляр.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 18.02.2019
 */
public class LbTopicReaderService extends AbstractService {
    private static final Logger log = LogManager.getLogger();
    private static final Logger queuesStateLog = LogManager.getLogger("queuesState");

    private static final CompletableFuture<?>[] EMPTY_COMPLETABLE_FUTURE_ARRAY = new CompletableFuture<?>[0];

    private final TopicId topicId;

    private final SingleThreadExecutorServiceFactory executorServiceFactory;
    private final LbApiStreamConsumerFactory streamConsumerFactory;
    private final PartitionDao partitionDao;
    private final ReadSemaphore readSemaphore;
    private final LogBrokerPartitionSourceContextsStorage sourceContextsStorage;
    private final LogShatterService logShatterService;

    private final CompressionRatioCalculator compressionRatioCalculator = new CompressionRatioCalculator();

    private volatile ExecutorService executorService;
    private volatile StreamConsumer streamConsumer;

    LbTopicReaderService(
        TopicId topicId,
        SingleThreadExecutorServiceFactory executorServiceFactory,
        LbApiStreamConsumerFactory streamConsumerFactory,
        PartitionDao partitionDao,
        ReadSemaphore readSemaphore,
        LogBrokerPartitionSourceContextsStorage sourceContextsStorage,
        LogShatterService logShatterService
    ) {
        this.topicId = topicId;
        this.executorServiceFactory = executorServiceFactory;
        this.streamConsumerFactory = streamConsumerFactory;
        this.partitionDao = partitionDao;
        this.readSemaphore = readSemaphore;
        this.sourceContextsStorage = sourceContextsStorage;
        this.logShatterService = logShatterService;
    }

    public TopicId getTopicId() {
        return topicId;
    }

    public Optional<Double> getCompressionRatio() {
        return compressionRatioCalculator.canCalculatePreciseCompressionRatio()
            ? Optional.of(compressionRatioCalculator.getCompressionRatio())
            : Optional.empty();
    }

    public void logLogBatchQueueStates() {
        logLogBatchQueueState("ParseQueue", SourceContext::getCreationTimeOfTheLastBatchInParseQueue);
        logLogBatchQueueState("OutputQueue", SourceContext::getCreationTimeOfTheLastBatchInOutputQueue);
    }

    private void logLogBatchQueueState(String queueName, Function<SourceContext, Long> sourceContextToCreationTimeOfTheLastBatchFunction) {
        List<ImmutablePair<LogbrokerSourceContext, Long>> contextAgePairs =
            sourceContextsStorage.getSourceContextsWithOldestLogBatches(sourceContextToCreationTimeOfTheLastBatchFunction);

        if (!contextAgePairs.isEmpty()) {
            long currentTimeMillis = System.currentTimeMillis();

            queuesStateLog.info("Source contexts with oldest batches in {} for topic {}:", queueName, topicId.asString());
            for (ImmutablePair<LogbrokerSourceContext, Long> pair : contextAgePairs) {
                queuesStateLog.info(
                    "{}s {}",
                    TimeUnit.MILLISECONDS.toSeconds(currentTimeMillis - pair.getRight()),
                    pair.getLeft()
                );
            }
        }
    }

    @Override
    protected void doStart() {
        // Это поток, который ищет/создаёт SourceContext'ы и добавляет данные в очередь на парсинг. Он один на топик,
        // что то же самое что один поток на сессию чтения. Много потоков сделать нетривиально, потому что тогда
        // сообщения от Логброкера будут перемешиваться, и это сломает всё и везде. Если будет не хватать одного потока,
        // то можно наверное придумать какую-нибудь схему с делением партиций на группы и выделением потока на группы
        // партиций. Ну или ещё довольно просто было бы сделать по потоку на партицию, но партиций наверное слишком
        // много для этого.
        executorService = executorServiceFactory.create("topic_" + topicId.asString());

        executorService.submit(() -> {
            try {
                log.info("Creating StreamConsumer for topic '{}'", topicId.asString());
                streamConsumer = streamConsumerFactory.createStreamConsumer(topicId, executorService);

                log.info("Starting StreamConsumer for topic '{}'", topicId.asString());
                streamConsumer.startConsume(new StreamListenerImpl());
            } catch (Throwable throwable) {
                log.error("Failed to start session for topic '" + topicId.asString() + "'", throwable);
                notifyFailed(throwable);
                cleanUp();
            }
        });
    }

    @Override
    protected void doStop() {
        executorService.submit(() -> {
            cleanUp();
            notifyStopped();
        });
    }

    private void cleanUp() {
        if (streamConsumer != null) {
            try {
                log.info("Stopping StreamConsumer for topic '{}'", topicId.asString());
                streamConsumer.stopConsume();
            } catch (Throwable throwable) {
                log.error("Failed to stop StreamConsumer for topic '" + topicId.asString() + "'", throwable);
                notifyFailed(throwable);
            }
        }

        // Останавливаем executorService последним потому что streamConsumer его использует
        if (executorService != null) {
            executorService.shutdown();
        }
    }


    private class StreamListenerImpl implements StreamListener {
        private volatile LbApiLogger logger = new LbApiLogger("_unknown_", topicId);

        /**
         * Вызывается один раз сразу после открытия соединения с Логброкером. Тут приходит id сессии, который полезно
         * сохранить и писать в лог.
         *
         * Этот метод вызывается из пула потоков GRPC, а не из {@link #executorService}.
         */
        @Override
        public void onInit(ConsumerInitResponse init) {
            logger = new LbApiLogger(init.getSessionId(), topicId);

            LbApiActionLogger actionLogger = logger.init();
            try {
                notifyStarted();

                actionLogger.success();
            } catch (Throwable throwable) {
                actionLogger.failure(throwable);
                onError(throwable);
            }
        }

        /**
         * Вызывается когда Логброкер предлагает этому инстансу Логшаттера читать эту партицию.
         *
         * API Логброкера не умеет отдавать список партиций. Зато оно умеет балансировать партиции между сессиями
         * одного клиента в рамках одного ДЦ. Когда Логброкер хочет чтобы сессия читала какую-то партицию, он отправляет
         * в эту сессию сообщение Lock. Это единственный правильный способ для Логшаттера узнать какие вообще партиции
         * есть. Больше об этом здесь: https://wiki.yandex-team.ru/logbroker/docs/protocols/grpc/#read-with-locks.
         *
         * Логброкер гарантирует что каждую партицию читает ровно одна сессия, но только в рамках одного ДЦ. Каждый
         * инстанс Логшаттера умеет ходить в каждый ДЦ Логброкера и читает только первородные партиции. Это значит что
         * синхронизацию между разными инстансами Логшаттера можно поручить Логброкеру.
         *
         * Когда приходит сообщение Lock с какой-то партицией от Логброкера, мы просто отвечаем что готовы читать.
         */
        @Override
        public void onLock(ConsumerLockMessage lock, BasicStreamListener.LockResponder lockResponder) {
            LbApiActionLogger actionLogger = logger.lock(lock);
            try {
                // Получаем оффсет из Монги, говорим Логброкеру что мы готовы читать эту партицию, передаём оффсет.
                // Если в Монге не было оффсета для этой партиции, то передаём 0.
                long offset = getPartitionOffset(lock).orElse(0L);

                lockResponder.locked(offset, false);

                actionLogger.success(builder -> builder.add("offset", offset));
            } catch (Throwable throwable) {
                actionLogger.failure(throwable);
                onError(throwable);
            }
        }

        private Optional<Long> getPartitionOffset(ConsumerLockMessage lock) {
            String partition = PartitionIdUtils.toString(lock.getTopic(), lock.getPartition());
            return Optional.ofNullable(partitionDao.get(partition))
                .map(LogBrokerOffset::getOffset)
                // Если партицию ещё никогда не читали, но таска получения оффсетов уже отработала, то в Монге может
                // быть оффсет -1. Нумерация сообщений в новом API начинается с нуля, если передать -1, то Логброкеру не
                // понравится. Делаем Math.max, потому что согласно https://nda.ya.ru/3UYbnd откатывать оффсет назад
                // нельзя в любом случае.
                .map(offset -> Math.max(offset, lock.getReadOffset()));
        }

        /**
         * Вызывается когда Логброкер хочет чтобы этот инстанс Логшаттера прекратил читать эту партицию.
         * После того, как пришёл Release, данные по этой партиции уже не придут.
         *
         * Это может происходить при перебалансировке партиций. Когда один инстанс Логшаттера лежит, все партиции будут
         * распределены между другими инстансами Логшаттера. Когда лежавший инстанс поднимется, Логброкер перенесёт
         * часть партиций на этот инстанс. Инстансам, с которых уходят партиции, придут сообщения Release.
         *
         * В {@link LbApiStreamConsumerFactory} выставляется setForceBalancePartitions(false). Это значит что после
         * Release Логброкер дождётся всех коммитов от сессии, которая получила Release прежде чем слать Lock и данные в
         * другую сессию.
         *
         * На Release можно не реагировать. Мы просто штатно допишем все данные, закоммитим оффсеты, и Логброкер начнёт
         * слать данные в другую сессию.
         *
         * Если идут ученьки или есть проблемы с сетевой связностью, то сессия упадёт, и Логброкер поймёт что коммитов
         * он уже не дождётся. При такой схеме могут возникнуть проблемы если какому-то инстансу Логшаттера доступен
         * Логброкер, но недоступен Кликхаус или Монга. В таком случае мы не сможем ничего закоммитить, и Логброкер
         * будет нас ждать. Вроде бы сессии падают если какое-то время в них нет никакой активности, но это не точно.
         * Решили что ситуация, когда Логброкер бесконечно ждёт коммитов между Release и Lock очень маловероятна.
         */
        @Override
        public void onRelease(ConsumerReleaseMessage release) {
            logger.release(release).success();
        }

        /**
         * Вызывается когда приходит пачка данных от Логброкера. В пачке могут быть логи из разных хостов и файлов.
         * Добавляем все данные в очередь парсинга.
         *
         * Не распаковываем, потому что это слишком медленно. У нас один поток на сессию чтения, что эквивалентно одному
         * потоку на топик. Медленные вещи здесь делать нельзя. Распаковкой и разбиением на строки занимаются
         * потоки-парсеры.
         *
         * После того, как все данные из пачки будут обработаны, сообщаем об этом Логброкеру.
         */
        @Override
        public void onRead(ConsumerReadResponse read, StreamListener.ReadResponder readResponder) {
            LbApiActionLogger actionLogger = logger.read(read);
            try {
                waitForParseQueueToHaveFreeSpace();

                AddMessagesToQueueResult addMessagesToQueueResult = addMessageBatchesToParseQueue(read.getBatches());

                // В отдельных переменных чтобы не тащить addMessagesToQueueResult в лямбду ниже и не держать его в
                // памяти до коммита. addMessagesToQueueResult должен стать мусором сразу после выхода из onRead.
                long cookie = read.getCookie();
                long skippedMessagesCount = addMessagesToQueueResult.skippedMessages.size();
                long skippedCompressedDataSizeBytes = addMessagesToQueueResult.skippedMessages.stream()
                    .mapToLong(message -> message.getRawData().length)
                    .sum();

                addMessagesToQueueResult.endOfProcessingFuture
                    .whenComplete((v, t) -> {
                        // Эта лямбда запускается в потоках, которые пишут в Кликхаус
                        // TODO дождаться ответа (пофиксить багу в клиенте ЛБ)
                        actionLogger.timedStage(
                            "commit",
                            builder -> builder
                                .add("cookie", cookie)
                                .add("skipped_messages_count", skippedMessagesCount)
                                .add("skipped_compressed_data_size", skippedCompressedDataSizeBytes)
                        );
                        readResponder.commit();
                    });
            } catch (Throwable throwable) {
                actionLogger.failure(throwable);
                onError(throwable);
            }
        }

        /**
         * Вызывается когда Логброкер сообщает что успешно закоммитил какие-то оффсеты. На это реагировать не нужно.
         */
        @Override
        public void onCommit(CommitMessage commit) {
        }

        /**
         * Вызывается когда сессия закрывается штатно, без ошибки. Это означает что Логброкер уже начал раздавать
         * партиции, которые читала эта сессия, другим Логшаттерам, и значит нужно как можно быстрее перестать писать в
         * Кликхаус и Монгу.
         *
         * Освобождаем все оставшиеся ресурсы и помечаем сервис {@link State#TERMINATED}, кто-то выше знает как
         * порестартить и вообще нужно ли.
         *
         * Этот метод вызывается из пула потоков GRPC, а не из {@link #executorService}.
         */
        @Override
        public void onClose() {
            LbApiActionLogger actionLogger = logger.close();
            try {
                stopProcessingAlreadyReadData();

                cleanUp();
                notifyStopped();

                actionLogger.success();
            } catch (Throwable throwable) {
                actionLogger.failure(throwable);
            }
        }

        /**
         * Вызывается когда сессия закрывается из-за того, что произошла какая-то ошибка. Это могут быть проблемы с
         * сетью или ошибка, которую прислал Логброкер. Как в в {@link #onClose()}, если мы сюда зашли, значит сессия
         * уже закрылась, и Логброкер уже раздаёт партиции другим Логшаттерам. Нужно как можно быстрее перестать писать
         * в Кликхаус и Монгу.
         *
         * Освобождаем все оставшиеся ресурсы и помечаем сервис {@link State#FAILED}, кто-то выше знает как
         * порестартить.
         *
         * Этот метод вызывается из пула потоков GRPC, а не из {@link #executorService}.
         */
        @Override
        public void onError(Throwable throwableFromLogBroker) {
            log.error("", throwableFromLogBroker);

            LbApiActionLogger actionLogger = logger.error(throwableFromLogBroker);
            try {
                stopProcessingAlreadyReadData();

                cleanUp();

                actionLogger.success();
            } catch (Throwable throwable) {
                actionLogger.failure(throwable);
            } finally {
                notifyFailed(throwableFromLogBroker);
            }
        }

        /**
         * Говорим всем SourceContext'ам что надо перестать парсить и писать в Кликхаус и Монгу. Уже начатые запросы к
         * Кликхаусу и Монге продолжатся, новые начинаться не будут, данные будут выброшены, в следующий раз прочитаем
         * их ещё раз.
         */
        private void stopProcessingAlreadyReadData() {
            sourceContextsStorage.finishAll();
        }

        /**
         * Ждём разрешение читать дальше от ReadSemaphore.
         * ReadSemaphore разрешит читать только когда очередь парсинга не переполнена и Логшаттер не завершает работу.
         * Это защита от OOM в случаях, когда чтение идёт быстрее чем парсинг и запись в Кликхаус. Тут небольшая гонка,
         * потому что мы проверяем что в очереди есть место и добавляем данные в очередь неатомарно. Это неважно, потому
         * что худшее что может случиться - это мы добавим в очередь на несколько ConsumerReadResponse'ов больше, чем в
         * неё влезает. Предполагается что размер одного ConsumerReadResponse на порядки меньше чем размер очереди. Для
         * разных топиков у нас разные сессии и разные потоки, поэтому можно спокойно ждать, на другие топики это не
         * повлиет.
         *
         * Здесь есть два вида очередей: глобальная очередь и отдельные очереди на иденты и топики. Нужно чтобы во всех
         * очередях было место.
         */
        private void waitForParseQueueToHaveFreeSpace() {
            try {
                // Ищем отдельные очереди для топика или идента
                ReadSemaphore.QueuesCounter queuesCounterForSource =
                    readSemaphore.getQueuesCounterForSource(topicId.asStringWithoutDataCenter());

                // Ждём свободного места в глобальной очереди как минимум один раз
                readSemaphore.waitForRead();

                // Если в отдельных очередях есть место, проходим дальше. Если нет места, то ждём и пробуем ещё раз.
                // Выйдем из этого цикла только когда дождёмся свободного места в глобальной очереди, и сразу после
                // этого будет место в отдельных очередях.
                while (queuesCounterForSource.getQueueThatReachedLimit() != null) {
                    Thread.sleep(100);
                    readSemaphore.waitForRead();
                }
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private AddMessagesToQueueResult addMessageBatchesToParseQueue(List<MessageBatch> messageBatches) {
        List<AddMessagesToQueueResult> addMessagesToQueueResults = messageBatches.stream()
            .map(this::addMessageBatchToParseQueue)
            .collect(Collectors.toList());

        return new AddMessagesToQueueResult(
            CompletableFuture.allOf(
                addMessagesToQueueResults.stream()
                    .map(result -> result.endOfProcessingFuture)
                    .toArray(CompletableFuture<?>[]::new)
            ),
            addMessagesToQueueResults.stream()
                .flatMap(addMessagesToQueueResult -> addMessagesToQueueResult.skippedMessages.stream())
                .collect(Collectors.toList())
        );
    }

    /**
     * @return CompletableFuture, которая закомплитится когда все данные будут обработаны и записаны, и список
     * сообщений, для которых не нашлось конфигов.
     */
    private AddMessagesToQueueResult addMessageBatchToParseQueue(MessageBatch batch) {
        // Говорим мониторингу про "слишком долго ничего не читали" что мы только что прочитали что-то.
        readSemaphore.notifyRead();

        List<MessageData> skippedMessages = new ArrayList<>();

        Multimap<LogbrokerSourceContext, MessageData> sourceContextToMessagesMap = HashMultimap.create();
        for (MessageData message : batch.getMessageData()) {
            List<LogbrokerSourceContext> sourceContexts = findSourceContexts(batch, message);
            if (sourceContexts.isEmpty()) {
                skippedMessages.add(message);
            } else {
                for (LogbrokerSourceContext sourceContext : sourceContexts) {
                    if (message.getMessageMeta().getSeqNo() <= sourceContext.getReaderSeqno()) {
                        // повторно читаем уже обработанный чанк, поэтому пропускаем
                        // Если несколько конфигов для одних данных, то можем посчитать одно сообщение как пропущенное
                        // несколько раз, это не страшно.
                        skippedMessages.add(message);
                        continue;
                    }
                    sourceContextToMessagesMap.put(sourceContext, message);
                }
            }
        }

        List<CompletableFuture<Void>> processedFutures = new ArrayList<>();

        sourceContextToMessagesMap.asMap().forEach(
            (sourceContext, messages) -> {
                // Этот код выполняется потоком логброкерной сессии. Там один поток на сесию, что означает один
                // поток на топик. Здесь нельзя распаковывать данные, потому что это будет слишком медленно.
                // Распаковкой занимаются потоки-парсеры.

                long offset = messages.stream()
                    .mapToLong(MessageData::getOffset)
                    .max().orElseThrow(() -> new RuntimeException("bug in addMessageBatchToParseQueue"));
                long seqNo = messages.stream()
                    .mapToLong(message -> message.getMessageMeta().getSeqNo())
                    .max().orElseThrow(() -> new RuntimeException("bug in addMessageBatchToParseQueue"));
                long compressedDataSize = messages.stream()
                    .mapToLong(message -> message.getRawData().length)
                    .sum();
                long approximateDecompressedDataSize =
                    (long) (compressionRatioCalculator.getCompressionRatio() * compressedDataSize);

                LogBatch logBatch = new LogBatch(
                    lazilyDecompressAndUpdateCompressionRatio(messages),
                    offset,
                    seqNo,
                    approximateDecompressedDataSize,
                    Duration.ZERO,
                    sourceContext.getLogParser().getTableDescription().getColumns(),
                    batch.getTopic() + ":" + batch.getPartition()
                );

                processedFutures.add(logBatch.getEndOfProcessingFuture());

                sourceContext.setReaderSeqno(seqNo);
                sourceContext.setReaderPartitionOffset(offset);

                sourceContext.getQueuesCounter().increment(approximateDecompressedDataSize);
                readSemaphore.incrementGlobalQueue(approximateDecompressedDataSize);

                sourceContext.getParseQueue().add(logBatch);
                logShatterService.addToParseQueue(sourceContext);
            }
        );

        return new AddMessagesToQueueResult(
            CompletableFuture.allOf(processedFutures.toArray(EMPTY_COMPLETABLE_FUTURE_ARRAY)),
            skippedMessages
        );
    }

    private List<LogbrokerSourceContext> findSourceContexts(MessageBatch batch, MessageData data) {
        return sourceContextsStorage.getSourceContexts(
            getLogBrokerSourceKey(batch, data),
            getInstanceId(data)
        );
    }

    private static LogBrokerSourceKey getLogBrokerSourceKey(MessageBatch batch, MessageData data) {
        TopicId topicId = TopicId.fromString(batch.getTopic());
        return new LogBrokerSourceKey(
            findFirstNonEmptyValueForKey(data.getMessageMeta().getExtraFields(), "server").orElse(null),
            findFirstNonEmptyValueForKey(data.getMessageMeta().getExtraFields(), "file").orElse(null),
            new String(data.getMessageMeta().getSourceId(), StandardCharsets.UTF_8),
            topicId.getIdent(),
            batch.getTopic(),
            String.valueOf(batch.getPartition()),
            topicId.getLogType()
        );
    }

    private static Optional<String> findFirstNonEmptyValueForKey(Map<String, List<String>> extraFields, String key) {
        return extraFields
            .getOrDefault(key, Collections.emptyList())
            .stream()
            .filter(string -> !Strings.isNullOrEmpty(string))
            .findFirst();
    }

    private static int getInstanceId(MessageData data) {
        try {
            return findFirstNonEmptyValueForKey(data.getMessageMeta().getExtraFields(), "x-slot")
                .map(Integer::parseInt)
                .orElse(0);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    /**
     * Не хотим держать в памяти распакованные данные больше чем нужно. Хотим распаковать и разбить на строки прямо
     * перед парсингом, и сразу после парсинга потерять все ссылки на распакованные данные. Сразу после распаковки
     * добавляем размеры запакованных и распакованных данных в {@link #compressionRatioCalculator}.
     */
    private Stream<String> lazilyDecompressAndUpdateCompressionRatio(Collection<MessageData> messages) {
        return messages.stream()
            .flatMap(message -> {
                BufferedReader bufferedReader = new BufferedReader(
                    new InputStreamReader(
                        new CompressionRatioUpdatingInputStream(
                            compressionRatioCalculator,
                            message.getRawData().length,
                            decompressData(
                                message.getMessageMeta().getCodec(),
                                message.getRawData()
                            )
                        ),
                        StandardCharsets.UTF_8
                    )
                );
                return bufferedReader
                    .lines()
                    .onClose(() -> {
                        try {
                            bufferedReader.close();
                        } catch (IOException e) {
                            throw new UncheckedIOException(e);
                        }
                    });
            });
    }

    private static InputStream decompressData(CompressionCodec compressionCodec, byte[] compressedData) {
        try {
            switch (compressionCodec) {
                case RAW:
                    return new ByteArrayInputStream(compressedData);
                case GZIP:
                    return new GZIPInputStream(new ByteArrayInputStream(compressedData));
                case LZOP:
                    return new LzopInputStream(new ByteArrayInputStream(compressedData));
                default:
                    throw new UnsupportedCodecException(compressionCodec);
            }
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }


    static class AddMessagesToQueueResult {
        final CompletableFuture<Void> endOfProcessingFuture;
        final List<MessageData> skippedMessages;

        AddMessagesToQueueResult(CompletableFuture<Void> endOfProcessingFuture, List<MessageData> skippedMessages) {
            this.endOfProcessingFuture = endOfProcessingFuture;
            this.skippedMessages = skippedMessages;
        }
    }

    static class CompressionRatioUpdatingInputStream extends FilterInputStream {
        private final CompressionRatioCalculator compressionRatioCalculator;
        private final long compressedSizeBytes;
        private final CountingInputStream countingInputStream;

        CompressionRatioUpdatingInputStream(CompressionRatioCalculator compressionRatioCalculator,
                                            long compressedSizeBytes, InputStream in) {
            this(compressionRatioCalculator, compressedSizeBytes, new CountingInputStream(in));
        }

        private CompressionRatioUpdatingInputStream(CompressionRatioCalculator compressionRatioCalculator,
                                                    long compressedSizeBytes, CountingInputStream countingInputStream) {
            super(countingInputStream);
            this.countingInputStream = countingInputStream;
            this.compressionRatioCalculator = compressionRatioCalculator;
            this.compressedSizeBytes = compressedSizeBytes;
        }

        @Override
        public void close() throws IOException {
            compressionRatioCalculator.add(compressedSizeBytes, countingInputStream.getCount());
            super.close();
        }
    }
}
