package ru.yandex.market.logshatter.reader.startrek;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.log4j.Logger;
import org.joda.time.Instant;
import ru.yandex.startrek.client.model.Issue;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

class SimpleClient {
    private static final Logger log = Logger.getLogger(SimpleClient.class);

    private static final Gson GSON;

    static {
        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(SimpleHistoryEvent.class, new SimpleEventDeserializer());
        GSON = gsonBuilder.create();
    }

    private final String apiUrl;
    private final String accessToken;

    SimpleClient(String apiUrl, String accessToken) {
        this.apiUrl = apiUrl + "/v2/issues/";
        this.accessToken = accessToken;
    }

    List<SimpleHistoryEvent> getSimpleEvents(Issue issue, Set<String> eventKeys) {
        String changesUrl = apiUrl + issue.getKey() + "/changelog";

        if (!eventKeys.isEmpty()) {
            changesUrl += "?field=" + String.join(",", eventKeys);
        }

        HttpClient client = HttpClientBuilder.create().build();
        HttpGet get = new HttpGet(changesUrl);
        get.addHeader("Authorization", "OAuth " + accessToken);
        try {
            log.info("GET " + changesUrl);
            HttpResponse response = client.execute(get);
            StatusLine statusLine = response.getStatusLine();
            if (statusLine.getStatusCode() != HttpStatus.SC_OK) {
                String errorMessage = "Changes response " + changesUrl +
                    " failed with code " + statusLine.getStatusCode() +
                    ". Reason: " + statusLine.getReasonPhrase();

                throw new IllegalStateException(errorMessage);
            }

            Reader responseReader = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));
            return GSON.fromJson(responseReader, new TypeToken<List<SimpleHistoryEvent>>() {
            }.getType());
        } catch (IOException e) {
            throw new RuntimeException("Error on changes loading", e);
        }

    }

    static class SimpleHistoryEvent {
        private long eventMillis;
        private String updatedBy;
        private Map<String, String> fieldChanges = new HashMap<>();

        long getEventMillis() {
            return eventMillis;
        }

        String getUpdatedBy() {
            return updatedBy;
        }

        Map<String, String> getFieldChangesMap() {
            return fieldChanges;
        }
    }

    static class SimpleEventDeserializer implements JsonDeserializer<SimpleHistoryEvent> {

        @Override
        public SimpleHistoryEvent deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
            throws JsonParseException {

            JsonObject eventJson = json.getAsJsonObject();
            SimpleHistoryEvent simpleHistoryEvent = new SimpleHistoryEvent();
            simpleHistoryEvent.eventMillis = Instant.parse(eventJson.get("updatedAt").getAsString()).getMillis();
            simpleHistoryEvent.updatedBy = eventJson.get("updatedBy").getAsJsonObject().get("id").getAsString();

            for (JsonElement change : eventJson.getAsJsonArray("fields")) {
                String field = change.getAsJsonObject().get("field").getAsJsonObject().get("id").getAsString();
                JsonElement toValueElement = change.getAsJsonObject().get("to");

                String newValue;
                if (toValueElement.isJsonNull() || toValueElement.isJsonArray()) {
                    continue;
                } else if (toValueElement.isJsonObject()) {
                    String valueKey = getKey(toValueElement);
                    if (valueKey == null) {
                        Set<String> keys = toValueElement.getAsJsonObject().keySet();
                        if (keys.isEmpty()) {
                            continue;
                        }
                        newValue = keys.iterator().next();
                    } else {
                        newValue = toValueElement.getAsJsonObject().get(valueKey).getAsString();
                    }
                } else {
                    newValue = toValueElement.getAsString();
                }

                simpleHistoryEvent.fieldChanges.put(field, newValue);
            }

            return simpleHistoryEvent;
        }

        private String getKey(JsonElement toValueElement) {
            if (toValueElement.getAsJsonObject().has("key")) {
                return "key";
            }
            if (toValueElement.getAsJsonObject().has("id")) {
                return "id";
            }
            return null;
        }
    }
}
