package ru.yandex.market.logshatter.reader.startrek;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.SearchRequest;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

/**
 * @author imelnikov
 */
public class StartrekClient {
    private static final int MAX_CONNECTIONS = 10;
    private static final int CONNECTION_TIMEOUT_SECONDS = 15;
    private static final int SOCKET_TIMEOUT_SECONDS = 10;

    private String apiUrl;

    private SimpleClient simpleClient;

    private final String accessToken;

    public StartrekClient(String accessToken) {
        this.accessToken = accessToken;
    }

    Stream<Issue> getIssues(String queue, Date date, int limit) {
        Session session = getSession();

        //StarTrack filters not supported hours
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        StringBuilder filters = new StringBuilder("Queue: ").append(queue);

        if (date.getTime() > 0) {
            filters.append(" Updated: > \"").append(dateFormat.format(date)).append("\"");
        }

        filters.append(" \"Sort By\": Updated");

        SearchRequest query = new SearchRequest.Builder()
            .query(filters.toString())
            .build();

        return session.issues().find(query)
            .take(limit)
            .toList()
            .stream()
            .filter(issue -> issue.getUpdatedAt().getMillis() > date.getTime());
    }

    /**
     * https://wiki.yandex-team.ru/tracker/api/issues/changelog/
     */
    List<StarTrekHistoryEvent> getKeyEvents(Issue issue, Set<String> eventKeys) {
        final List<StarTrekHistoryEvent> history = new ArrayList<>();

        for (SimpleClient.SimpleHistoryEvent simpleHistoryEvent : simpleClient.getSimpleEvents(issue, eventKeys)) {
            for (Map.Entry<String, String> changes : simpleHistoryEvent.getFieldChangesMap().entrySet()) {
                if (!eventKeys.contains(changes.getKey())) {
                    continue;
                }

                history.add(new StarTrekHistoryEvent(
                    simpleHistoryEvent.getEventMillis(),
                    changes.getKey(),
                    changes.getValue(),
                    simpleHistoryEvent.getUpdatedBy()));
            }
        }

        return history;
    }

    protected Session getSession() {
        final MapF<String, Field.Schema> customFields = Cf.<String, Field.Schema>hashMap()
            .plus1("sreBeginTime", Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
            .plus1("sreEndTime", Field.Schema.scalar(Field.Schema.Type.DATETIME, false));

        return StartrekClientBuilder.newBuilder()
            .uri(apiUrl)
            .maxConnections(MAX_CONNECTIONS)
            .connectionTimeout(CONNECTION_TIMEOUT_SECONDS, TimeUnit.SECONDS)
            .socketTimeout(SOCKET_TIMEOUT_SECONDS, TimeUnit.SECONDS)
            .customFields(customFields)
            .build(accessToken);
    }

    public void setApiUrl(String apiUrl) {
        this.apiUrl = apiUrl;
        simpleClient = new SimpleClient(apiUrl, accessToken);
    }
}
