package ru.yandex.market.logshatter.reader.startrek;

import com.google.common.collect.Multimap;
import com.google.gson.Gson;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.InitializingBean;
import ru.yandex.bolts.collection.Option;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.config.LogSource;
import ru.yandex.market.logshatter.reader.AbstractReaderService;
import ru.yandex.market.monitoring.MonitoringUnit;
import ru.yandex.startrek.client.model.Component;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.UserRef;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/*
 * Startrek reader
 *
 * Read issues from specified queues and log stats
 * Use URI startrek://QUEUENAME
 *
 * @author imelnikov
 */
public class StartrekReaderService extends AbstractReaderService implements InitializingBean {

    private static final String STARTREK_SCHEMA = "startrek";

    private StartrekClient startrekClient;

    private MonitoringUnit monitoringUnit = new MonitoringUnit("StartrekReaderService");
    private List<StartrekSourceContext> sourceContexts = new ArrayList<>();

    private final static List<String> skipFieldsRules = Arrays.asList("internalDesign", "security");
    private static final Set<String> eventKeys = new HashSet<>(Collections.singletonList("status"));

    private Collection<String> queuesWhitelist;

    private int batchSize = 100;
    private int sleepTimeoutMinutes = 10;
    private int firstLoadDays = 90;
    private int issuesLimit = 1000;

    @Override
    public void afterPropertiesSet() throws Exception {
        Multimap<LogSource, LogShatterConfig> tables = configurationService.getConfigsForSchema(STARTREK_SCHEMA);
        if (tables.isEmpty()) {
            log.info("No startrek configs. StartrekReaderService stops.");
            return;
        }

        for (Map.Entry<LogSource, Collection<LogShatterConfig>> tableEntry : tables.asMap().entrySet()) {
            LogSource source = tableEntry.getKey();
            for (LogShatterConfig config : tableEntry.getValue()) {
                if (queuesWhitelist.isEmpty() || queuesWhitelist.contains(source.getPath())) {
                    sourceContexts.add(
                        new StartrekSourceContext(config, source, errorLoggerFactory, readSemaphore.getEmptyQueuesCounter())
                    );
                } else {
                    log.info("Ignoring startrek source {}", source);
                }
            }
        }

        monitoring.getHostCritical().addUnit(monitoringUnit);

        new Thread(() -> {
            log.info("Starting StartrekReaderService");
            while (!Thread.interrupted() && isRunning()) {
                log.info("Trying to get leaderLatch");
                try (LeaderLatch leaderLatch = readSemaphore.getDistributedLock("startrek", null)) {
                    log.info("Awaiting 1 second for leadership. Has leadership: " + leaderLatch.hasLeadership());
                    leaderLatch.await(1, TimeUnit.SECONDS);
                    while (!Thread.interrupted() && isRunning()) {
                        try {
                            log.info("Awake, has leadership: " + leaderLatch.hasLeadership());
                            if (leaderLatch.hasLeadership()) {
                                log.info("Starting to process StQueues");
                                processStQueues();
                            } else {
                                monitoringUnit.ok();
                            }
                            log.info("Go to sleep for " + sleepTimeoutMinutes + " minutes");
                            TimeUnit.MINUTES.sleep(sleepTimeoutMinutes);
                        } catch (Exception e) {
                            log.error("Failed to load data from startrek", e);
                        }
                    }
                } catch (Exception e) {
                    log.error("Failed to get leaderLatch", e);
                }
            }
        }).start();
    }

    // Load data from each queue
    private void processStQueues() {
        StringBuilder failedQueues = new StringBuilder();
        for (StartrekSourceContext ctx : sourceContexts) {
            try {
                loadIssues(ctx);
            } catch (Exception e) {
                String queue = ctx.getSourceKey().getId();
                failedQueues.append(queue).append(" ");
                log.warn("Failed to load issues from queue " + queue, e);
            }
        }

        if (failedQueues.length() != 0) {
            monitoringUnit.warning("Failed to load issues from queues: " + failedQueues.toString());
        } else {
            monitoringUnit.ok();
        }
    }

    private void loadIssues(StartrekSourceContext sourceContext) {
        logshatterMetaDao.load(sourceContext);

        AtomicLong readStartMillis = new AtomicLong(System.currentTimeMillis());

        List<String> lines = new ArrayList<>();

        String queue = sourceContext.getSourceKey().getId();
        long readerLastTimestamp = sourceContext.getReaderLastTimestamp();
        if (sourceContext.getDataOffset() >= readerLastTimestamp) {
            readerLastTimestamp = sourceContext.getDataOffset();
        }
        if (readerLastTimestamp == 0L) { // first init
            readerLastTimestamp = firstLoadDays < 0 ? 0 : System.currentTimeMillis() - TimeUnit.DAYS.toMillis(firstLoadDays);
        }
        AtomicLong lastTimestamp = new AtomicLong(readerLastTimestamp);
        Date date = new Date(lastTimestamp.get());
        log.info("last update for queue " + queue + " is " + date);

        startrekClient.getIssues(queue, date, issuesLimit)
            .filter(issue -> skipFieldsRules.stream().noneMatch(rule -> issue.getO(rule).isPresent()))
            .forEach(issue -> {
                final StarTrekLine starTrekLine = convertIssue(issue, date);
                lastTimestamp.set(starTrekLine.getUpdated());

                final String line = new Gson().toJson(starTrekLine);

                if (line != null) {
                    lines.add(line);
                }
                if (lines.size() > batchSize) {
                    createBatch(new ArrayList<>(lines), lastTimestamp.get(), readStartMillis.get(), sourceContext);
                    lines.clear();
                    try {
                        readSemaphore.waitForRead();
                    } catch (InterruptedException ignored) {
                    }
                    readStartMillis.set(System.currentTimeMillis());
                }
            });

        createBatch(lines, lastTimestamp.get(), readStartMillis.get(), sourceContext);
        sourceContext.setReaderLastTimestamp(lastTimestamp.get());

        // MARKETINFRA-3825 Зажигаем мониторинг если последний тикет был слишком давно
        Instant lastIssueInstant = new Instant(lastTimestamp.get());
        if (lastIssueInstant.isBefore(Instant.now().minus(Duration.standardDays(7)))) {
            throw new RuntimeException(String.format(
                "Last timestamp for queue %s is too old (%s). Perhaps there is no access to queue.",
                sourceContext.getName(), lastIssueInstant
            ));
        }
    }

    private StarTrekLine convertIssue(Issue issue, Date lastParsingDate) {
        final StarTrekLine starTrekLine = new StarTrekLine();
        starTrekLine.setLastParsingMillis(lastParsingDate.getTime());

        starTrekLine.setCreated(issue.getCreatedAt().getMillis());
        starTrekLine.setUpdated(issue.getUpdatedAt().getMillis());
        starTrekLine.setResolved(issue.getResolvedAt().map(Instant::getMillis).getOrElse(0L));
        starTrekLine.setQueue(issue.getQueue().getKey());

        starTrekLine.setComponents(issue.getComponents().stream()
            .map(ComponentRef::load)
            .map(Component::getName)
            .collect(Collectors.toList()));

        starTrekLine.setIssue(issue.getKey());
        starTrekLine.setType(issue.getType().getKey());
        starTrekLine.setPriority(issue.getPriority().getKey());
        starTrekLine.setCreatedBy(issue.getCreatedBy().getLogin());
        starTrekLine.setAssignee(issue.getAssignee().map(UserRef::getLogin).getOrElse(""));
        starTrekLine.setStatus(issue.getStatus().getKey());
        starTrekLine.setTags(issue.getTags());
        starTrekLine.setTestScope(issue.getO("testScope").isPresent());

        if (issue.getProject().isPresent()) {
            starTrekLine.setOptional(StarTrekLine.OptField.PROJECT, issue.getProject().get().getDisplay());
        }

        if (issue.getResolver().isPresent()) {
            starTrekLine.setOptional(StarTrekLine.OptField.RESOLVER, issue.getResolver().get().getLogin());
        }

        if (issue.getResolution().isPresent()) {
            starTrekLine.setOptional(StarTrekLine.OptField.RESOLUTION, issue.getResolution().get().getKey());
        }

        starTrekLine.<String>setOptional(StarTrekLine.OptField.STAND, getOptionalOrNull(issue, "stand"));
        starTrekLine.<Float>setOptional(StarTrekLine.OptField.WEIGHT, getOptionalOrNull(issue, "weight"));
        starTrekLine.<String>setOptional(StarTrekLine.OptField.ISSUE_WEIGHT, getOptionalOrNull(issue, "issueWeight"));
        starTrekLine.setOptional(StarTrekLine.OptField.SRE_BEGIN_TIME, getOptionalOrNull(issue, "sreBeginTime", Instant::getMillis));
        starTrekLine.setOptional(StarTrekLine.OptField.SRE_NOTIFICATION_TIME, getOptionalOrNull(issue, "notificationTime", Instant::getMillis));
        starTrekLine.setOptional(StarTrekLine.OptField.SRE_END_TIME, getOptionalOrNull(issue, "sreEndTime", Instant::getMillis));

        starTrekLine.setHistory(startrekClient.getKeyEvents(issue, eventKeys));

        return starTrekLine;
    }

    private <T> T getOptionalOrNull(Issue issue, String field) {
        return issue.getO(field).isPresent() ? issue.<Option.Some<T>>get(field).get() : null;
    }

    private <T, U> T getOptionalOrNull(Issue issue, String field, Function<U, T> extractFunction) {
        U value = getOptionalOrNull(issue, field);
        return value == null ? null : extractFunction.apply(value);
    }

    private void createBatch(List<String> lines, long lastTimestamp,
                             long readStartMillis, StartrekSourceContext context) {
        long readEndMillis = System.currentTimeMillis();
        long batchSizeBytes = LogShatterUtil.calcSizeBytes(lines);
        LogBatch logBatch = new LogBatch(
            lines.stream(),
            lastTimestamp,
            0,
            batchSizeBytes,
            java.time.Duration.ofMillis(readEndMillis - readStartMillis),
            context.getLogParser().getTableDescription().getColumns(),
            context.getName()
        );
        readSemaphore.incrementGlobalQueue(logBatch.getBatchSizeBytes());

        context.getParseQueue().add(logBatch);
        addToParseQueue(context);
        context.setDataOffset(lastTimestamp);
    }

    public void setStartrekClient(StartrekClient startrekClient) {
        this.startrekClient = startrekClient;
    }

    public void setSleepTimeoutMinutes(int sleepTimeoutMinutes) {
        this.sleepTimeoutMinutes = sleepTimeoutMinutes;
    }

    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    public void setFirstLoadDays(int firstLoadDays) {
        this.firstLoadDays = firstLoadDays;
    }

    public void setQueuesWhitelist(CharSequence queuesWhitelist) {
        this.queuesWhitelist = Pattern.compile(",")
            .splitAsStream(queuesWhitelist)
            .collect(Collectors.toSet());
    }

    public void setIssuesLimit(int issuesLimit) {
        this.issuesLimit = issuesLimit;
    }
}
