package ru.yandex.market.logshatter.rotation;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import ru.yandex.market.clickhouse.ClickHouseSource;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.LogshatterRestController;
import ru.yandex.market.logshatter.config.ConfigurationService;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Component
public class DataRotationService implements InitializingBean {

    private static final Logger log = LogManager.getLogger();
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyyMM");

    private final ClickhouseTemplate clickhouseTemplate;
    private final MonitoringUnit monitoringUnit = new MonitoringUnit("DataRotation");
    private final ClickHouseSource clickHouseSource;
    private final List<ArchiveSettings> archiveSettingsList;
    private final LeaderLatch leaderLatch;

    @Value("${logshatter.data-rotation.retry-pause-minutes}")
    private int retryPauseMinutes = 10;
    @Value("${logshatter.http.port}")
    private String httpPort;

    @Autowired
    public DataRotationService(LogShatterMonitoring monitoring, ConfigurationService configurationService,
                               ClickhouseTemplate clickhouseTemplate, ClickHouseSource clickHouseSource,
                               CuratorFramework curatorFramework) throws UnknownHostException {
        monitoring.getHostCritical().addUnit(monitoringUnit);
        monitoringUnit.setLogEnabled(true);
        this.clickhouseTemplate = clickhouseTemplate;
        this.clickHouseSource = clickHouseSource;
        this.archiveSettingsList = toArchiveSettings(configurationService.getConfigs());

        this.leaderLatch = new LeaderLatch(
            curatorFramework, "/data-rotation", InetAddress.getLocalHost().getCanonicalHostName()
        );
    }

    static int calcMaxPartition(Date fromDate, int daysInPast) {
        if (daysInPast == 0) {
            return 0;
        }

        final Calendar calendar = Calendar.getInstance();
        calendar.setTime(fromDate);
        calendar.add(Calendar.DAY_OF_MONTH, -daysInPast);
        return Integer.parseInt(DATE_FORMAT.format(calendar.getTime()));
    }

    private void addObsoletePartitions(List<ObsoletePartition> obsoletePartitions, String tableName, int maxPartition) {
        final List<String> hosts = clickHouseSource.isDistributed()
            ? new ArrayList<>(clickHouseSource.getShard2Hosts().values())
            : Collections.singletonList(clickHouseSource.getHost());


        final String[] tableNameParts = tableName.split("\\.");
        final String database = tableNameParts[0];
        final String table = tableNameParts[1];
        final String query = "SELECT DISTINCT partition" +
            " FROM system.parts" +
            " WHERE database = '" + database + "'" +
            " AND active" +
            " AND table = '" + table + "'" +
            " AND partition < '" + maxPartition + "'";

        for (String host : hosts) {
            clickhouseTemplate.query(
                query,
                host,
                rs -> obsoletePartitions.add(new ObsoletePartition(host, tableName, rs.getString("partition")))
            );
        }
    }

    @Scheduled(
        fixedDelayString = "#{${logshatter.data-rotation.days-between-archiving} * 24 * 60 * 60 * 1000}",
        initialDelayString = "#{${logshatter.data-rotation.run-delay-minutes} * 60 * 1000}"
    )
    public void runIteration() {
        while (!Thread.interrupted()) {
            try {
                if (!leaderLatch.hasLeadership()) {
                    log.info("Not leader, current leader is " + leaderLatch.getLeader().getId());
                    monitoringUnit.ok("Not leader");
                    return;
                }

                log.info("Starting refresh data rotation process");
                final List<ObsoletePartition> obsoletePartitions = getObsoletePartitions();

                if (obsoletePartitions.isEmpty()) {
                    monitoringUnit.ok();
                } else {
                    final String host = InetAddress.getLocalHost().getCanonicalHostName();
                    final String link = host + ":" + httpPort + LogshatterRestController.DROP_PARTITIONS_METHOD;
                    monitoringUnit.warning("Some partitions must be removed. Follow link " + link +
                        " to get database scripts");
                }

                break;
            } catch (Exception e) {
                monitoringUnit.warning("Refresh attempt failed", e);
                try {
                    TimeUnit.MINUTES.sleep(retryPauseMinutes);
                } catch (InterruptedException ie) {
                    log.error("Interrupted", ie);
                }
            }
        }
    }

    private List<ObsoletePartition> getObsoletePartitions() {
        final List<ObsoletePartition> obsoletePartitions = new ArrayList<>();
        final Date now = new Date();

        for (ArchiveSettings archiveSettings : archiveSettingsList) {
            final int maxPartition = calcMaxPartition(now, archiveSettings.dataRotationDays);
            addObsoletePartitions(obsoletePartitions, archiveSettings.tableName, maxPartition);
        }
        return obsoletePartitions;
    }

    public Multimap<String, ObsoletePartition> getObsoletePartitionsByHostMap() {


        final Multimap<String, ObsoletePartition> obsoletePartitionsByHostMap = ArrayListMultimap.create();
        getObsoletePartitions().forEach(p -> obsoletePartitionsByHostMap.put(p.getHost(), p));

        return obsoletePartitionsByHostMap;
    }

    public static List<ArchiveSettings> toArchiveSettings(List<LogShatterConfig> logShatterConfigs) {
        final Map<String, ArchiveSettings> tablesSettingsMap = new HashMap<>();
        final Set<String> tablesWithIncorrectSettings = new HashSet<>();

        for (LogShatterConfig logShatterConfig : logShatterConfigs) {
            if (logShatterConfig.getDataRotationDays() == 0) {
                continue;
            }

            final String localClickHouseTable = logShatterConfig.getInsertTableName();
            final int dataRotationDays = logShatterConfig.getDataRotationDays();

            final ArchiveSettings archiveSettings = tablesSettingsMap.getOrDefault(localClickHouseTable,
                new ArchiveSettings(localClickHouseTable, dataRotationDays));

            if (archiveSettings.dataRotationDays != dataRotationDays) {
                tablesWithIncorrectSettings.add(localClickHouseTable);
            }
            tablesSettingsMap.put(localClickHouseTable, archiveSettings);
        }

        if (!tablesWithIncorrectSettings.isEmpty()) {
            throw new RuntimeException("Few tables have different rotation settings: " +
                String.join(",", tablesWithIncorrectSettings));
        }

        return new ArrayList<>(tablesSettingsMap.values());
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        leaderLatch.start();
    }

    public static class ArchiveSettings {
        final String tableName;
        final int dataRotationDays;

        ArchiveSettings(String tableName, int dataRotationDays) {
            this.tableName = tableName;
            this.dataRotationDays = dataRotationDays;
        }

        public String getTableName() {
            return tableName;
        }

        public int getDataRotationDays() {
            return dataRotationDays;
        }
    }
}
