package ru.yandex.market.logshatter.spring;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.library.ticket_parser2.TvmApiSettings;
import ru.yandex.library.ticket_parser2.TvmClient;
import ru.yandex.market.logbroker.pull.LogBrokerClient;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.LogShatterService;
import ru.yandex.market.logshatter.config.ConfigurationService;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.meta.LogshatterMetaDao;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.logbroker.LogBrokerConfigurationService;
import ru.yandex.market.logshatter.reader.logbroker.LogBrokerReaderService;
import ru.yandex.market.logshatter.reader.logbroker.MonitoringLagThreshold;
import ru.yandex.market.logshatter.reader.logbroker.OldApiLogBrokerClients;
import ru.yandex.market.logshatter.reader.logbroker.PartitionDao;
import ru.yandex.market.logshatter.reader.logbroker.PartitionManager;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerLagLogger;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerLagMonitoring;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerLagWriterOldApi;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerSourcesWithoutMetadataMonitoring;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.MonitoringConfig;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.PartitionLeadersDebugLogger;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.PartitionLeadersLogger;
import ru.yandex.market.logshatter.reader.logbroker2.LogBrokerCleanupService;
import ru.yandex.market.logshatter.reader.logbroker2.LogBrokerReaderService2;
import ru.yandex.market.logshatter.reader.logbroker2.dc.LbDataCenterReaderServiceFactory;
import ru.yandex.market.logshatter.reader.logbroker2.threads.SingleThreadExecutorServiceFactoryImpl;
import ru.yandex.market.logshatter.reader.logbroker2.topic.LbApiStreamConsumerFactory;
import ru.yandex.market.logshatter.reader.logbroker2.topic.LbCredentialsProvider;
import ru.yandex.market.logshatter.reader.logbroker2.topic.LbTopicReaderServiceFactory;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 16/08/2018
 */
@Configuration
@EnableScheduling
public class LogshatterSpringLogbrokerConfig {
    private final Map<String, String> logBrokerClusterToHostMap;

    public LogshatterSpringLogbrokerConfig(@Value("${logshatter.logbroker.api.cluster-to-host}") String dataCenters) {
        Splitter commaSplitter = Splitter.on(',').trimResults().omitEmptyStrings();
        Splitter equalsSplitter = Splitter.on('=').trimResults().omitEmptyStrings().limit(2);
        this.logBrokerClusterToHostMap =
            commaSplitter.splitToList(dataCenters).stream()
                .map(equalsSplitter::splitToList)
                .collect(Collectors.toMap(
                    pair -> pair.get(0),
                    pair -> pair.get(1)
                ));
    }

    @Bean
    public PartitionDao partitionDao(@Value("${logshatter.mongo.main-url}") String mongoUrl,
                                     @Value("${logshatter.mongo.db-name:logshatter}") String dbName,
                                     @Value("${logshatter.mongo.connect-timeout-millis}") int connectTimeoutMillis,
                                     @Value("${logshatter.mongo.socket-timeout-millis}") int socketTimeoutMillis,
                                     @Value("${logshatter.mongo.ssl}") boolean ssl,
                                     @Value("${logshatter.mongo.replica-set}") String replicaSet) {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .connectTimeout(connectTimeoutMillis)
            .socketTimeout(socketTimeoutMillis)
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        return new PartitionDao(
            new MongoClient(
                new MongoClientURI(
                    mongoUrl,
                    options
                )
            )
                .getDatabase(dbName)
        );
    }

    @Bean(initMethod = "startAsync")
    public LogBrokerReaderService2 logBrokerReaderService(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LbDataCenterReaderServiceFactory dataCenterReaderServiceFactory
    ) {
        if (!newApi) {
            return null;
        }
        return new LogBrokerReaderService2(
            new SingleThreadExecutorServiceFactoryImpl(),
            dataCenterReaderServiceFactory,
            logBrokerClusterToHostMap.keySet()
        );
    }

    @Bean
    public LbDataCenterReaderServiceFactory dataCenterReaderServiceFactory(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LbTopicReaderServiceFactory topicReaderServiceFactory,
        @Autowired OldApiLogBrokerClients oldApiLogBrokerClients,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired PartitionDao partitionDao,
        @Autowired LogShatterMonitoring logShatterMonitoring,
        @Value("${logshatter.logbroker.offsets-fetching.retry-count-until-crit:5}")
            int offsetsFetchingRetryCountUntilCrit,
        @Value("${logshatter.logbroker.offsets-fetching.sleep-between-retries-millis:5000}")
            int offsetsFetchingSleepBetweenRetriesMillis
    ) {
        if (!newApi) {
            return null;
        }
        return new LbDataCenterReaderServiceFactory(
            topicReaderServiceFactory,
            new SingleThreadExecutorServiceFactoryImpl(),
            oldApiLogBrokerClients,
            logBrokerConfigurationService,
            partitionDao,
            logShatterMonitoring,
            offsetsFetchingRetryCountUntilCrit,
            offsetsFetchingSleepBetweenRetriesMillis
        );
    }


    @Bean
    public LbTopicReaderServiceFactory topicReaderServiceFactory(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LbApiStreamConsumerFactory streamConsumerFactory,
        @Autowired PartitionDao partitionDao,
        @Autowired ReadSemaphore readSemaphore,
        @Autowired LogShatterService logShatterService,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired LogshatterMetaDao logshatterMetaDao,
        @Autowired LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring,
        @Autowired BatchErrorLoggerFactory errorLoggerFactory
    ) {
        if (!newApi) {
            return null;
        }
        return new LbTopicReaderServiceFactory(
            new SingleThreadExecutorServiceFactoryImpl(),
            streamConsumerFactory,
            partitionDao,
            readSemaphore,
            logShatterService,
            logBrokerConfigurationService,
            logshatterMetaDao,
            sourcesWithoutMetadataMonitoring,
            errorLoggerFactory
        );
    }

    @Bean
    public LbApiStreamConsumerFactory streamConsumerFactory(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Value("${logshatter.logbroker.client}") String lbClient,
        @Autowired LbCredentialsProvider lbCredentialsProvider,
        @Value("${logshatter.logbroker.api.max-read-batch-size}") int maxReadBatchSize,
        @Value("${logshatter.logbroker.api.max-inflight-reads}") int maxInflightReads,
        @Value("${logshatter.logbroker.api.max-unconsumed-reads}") int maxUnconsumedReads
    ) {
        if (!newApi) {
            return null;
        }
        return new LbApiStreamConsumerFactory(
            logBrokerClusterToHostMap.entrySet().stream()
                .collect(Collectors.toMap(
                    entry -> entry.getKey(),
                    entry -> new LogbrokerClientFactory(new ProxyBalancer(entry.getValue()))
                )),
            lbCredentialsProvider,
            // Здесь имя клиента нужно в новом формате, со слешами. Дока ЛБ про это: https://nda.ya.ru/3UX7Aj.
            lbClient.replace('@', '/'),
            maxReadBatchSize,
            maxInflightReads,
            maxUnconsumedReads
        );
    }

    /**
     * logbroker.client-id - https://nda.ya.ru/3UXA3r
     * logshatter.{client-id,secret} - https://nda.ya.ru/3UXA3u
     */
    @Bean
    public LbCredentialsProvider credentialsProvider(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Value("${logshatter.logbroker.tvm.logbroker.client-id}") int logBrokerTvmClientId,
        @Value("${logshatter.logbroker.tvm.logshatter.client-id}") int logshatterTvmClientId,
        @Value("${logshatter.logbroker.tvm.logshatter.secret}") String logshatterTvmSecret
    ) {
        if (!newApi) {
            return null;
        }

        TvmApiSettings settings = TvmApiSettings.create();
        settings.setSelfClientId(logshatterTvmClientId);
        settings.enableServiceTicketsFetchOptions(logshatterTvmSecret, new int[]{logBrokerTvmClientId});
        TvmClient tvmClient = new TvmClient(settings);

        return () -> Credentials.tvm(tvmClient.getServiceTicketFor(logBrokerTvmClientId));
    }

    @Bean(initMethod = "afterPropertiesSet")
    public LogBrokerClient logbrokerClient(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Value("${logshatter.logbroker.host}") String lbHost,
        @Value("${logshatter.logbroker.client}") String lbClient,
        @Value("${dc}") String dc,
        @Value("${logshatter.logbroker.read-timeout-seconds}") int readTimeoutSeconds,
        @Value("${logshatter.logbroker.http-timeout-seconds}") int httpTimeoutSeconds,
        @Value("${logshatter.logbroker.session-timeout-seconds}") int sessionTimeoutSeconds
    ) throws Exception {
        if (newApi) {
            return null;
        }

        // Здесь имя клиента нужно в старом формате, слеши заменяются на @. Дока ЛБ про это: https://nda.ya.ru/3UX7Aj.
        LogBrokerClient logBrokerClient = new LogBrokerClient(
            lbHost,
            lbClient.replace('/', '@'),
            dc
        );
        logBrokerClient.setReadTimeoutSeconds(readTimeoutSeconds);
        logBrokerClient.setHttpTimeoutSeconds(httpTimeoutSeconds);
        logBrokerClient.setSessionTimeoutSeconds(sessionTimeoutSeconds);
        return logBrokerClient;
    }

    @Bean
    public OldApiLogBrokerClients oldApiLogBrokerClients(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Value("${logshatter.logbroker.host}") String lbHost,
        @Value("${logshatter.logbroker.client}") String lbClient,
        @Value("${logshatter.logbroker.read-timeout-seconds}") int readTimeoutSeconds,
        @Value("${logshatter.logbroker.http-timeout-seconds}") int httpTimeoutSeconds,
        @Value("${logshatter.logbroker.session-timeout-seconds}") int sessionTimeoutSeconds
    ) {
        if (!newApi) {
            return null;
        }
        return new OldApiLogBrokerClients(
            logBrokerClusterToHostMap.keySet().stream()
                .collect(Collectors.toMap(
                    dataCenter -> dataCenter,
                    dataCenter -> {
                        try {
                            LogBrokerClient logBrokerClient = new LogBrokerClient(
                                lbHost,
                                // Здесь имя клиента нужно в старом формате, слеши заменяются на @.
                                // Дока ЛБ про это: https://nda.ya.ru/3UX7Aj.
                                lbClient.replace('/', '@'),
                                dataCenter
                            );
                            logBrokerClient.setReadTimeoutSeconds(readTimeoutSeconds);
                            logBrokerClient.setHttpTimeoutSeconds(httpTimeoutSeconds);
                            logBrokerClient.setSessionTimeoutSeconds(sessionTimeoutSeconds);
                            logBrokerClient.afterPropertiesSet();
                            return logBrokerClient;
                        } catch (URISyntaxException | IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                ))
        );
    }

    @Bean
    public PartitionManager partitionManager(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LogShatterService logShatterService,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired ReadSemaphore readSemaphore,
        @Autowired LogshatterMetaDao logshatterMetaDao,
        @Autowired LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring,
        @Autowired BatchErrorLoggerFactory errorLoggerFactory,
        @Value("${logshatter.logbroker.max-acquired-partitions}") int maxAcquiredPartitions,
        @Value("${logshatter.logbroker.partitions-per-release-attempt}") int partitionsPerReleaseAttempt,
        @Value("${logshatter.http_properties.socket-timeout-seconds}") int clickhouseSocketTimeoutSeconds
    ) {
        if (newApi) {
            return null;
        }
        return new PartitionManager(
            logShatterService,
            logBrokerConfigurationService,
            readSemaphore,
            logshatterMetaDao,
            sourcesWithoutMetadataMonitoring,
            errorLoggerFactory,
            maxAcquiredPartitions,
            partitionsPerReleaseAttempt,
            clickhouseSocketTimeoutSeconds
        );
    }

    @Bean
    public MonitoringConfig monitoringConfig(
        @Autowired LogShatterMonitoring monitoring,
        @Value("${logshatter.logbroker.monitoring.lag-percent.default-warn}") double defaultWarnThresholdPercent,
        @Value("${logshatter.logbroker.monitoring.lag-percent.default-crit}") double defaultCritThresholdPercent,
        @Value("${logshatter.logbroker.monitoring.lag-percent.idents}") String identLagThresholds,
        @Value("${logshatter.logbroker.monitoring.delay-minutes}") int monitoringDelayMinutes,
        @Value("${logshatter.logbroker.sources-without-metadata.crit}") int sourcesWithoutMetadataForCrit
    ) {
        return new MonitoringConfig(
            monitoring,
            new MonitoringLagThreshold(defaultCritThresholdPercent, defaultCritThresholdPercent),
            parseIdentToLagThreshold(identLagThresholds),
            monitoringDelayMinutes, sourcesWithoutMetadataForCrit
        );
    }

    @VisibleForTesting
    static Map<String, MonitoringLagThreshold> parseIdentToLagThreshold(String identLagThresholds) {

        Map<String, MonitoringLagThreshold> identToLagThreshold = new HashMap<>();

        for (String identThreshold : Splitter.on(',').omitEmptyStrings().trimResults().split(identLagThresholds)) {
            List<String> identThresholdSplits = Splitter.on(':')
                .trimResults()
                .omitEmptyStrings()
                .splitToList(identThreshold);
            Preconditions.checkArgument(
                identThresholdSplits.size() == 3,
                "Invalid threshold config: '%s'. Format: " +
                    "'ident1:warn-threshold-rercent:crit-threshold-rercent," +
                    "ident2:warn-threshold-rercent:crit-threshold-rercent'",
                identLagThresholds
            );
            identToLagThreshold.put(
                identThresholdSplits.get(0),
                new MonitoringLagThreshold(
                    Double.parseDouble(identThresholdSplits.get(1)),
                    Double.parseDouble(identThresholdSplits.get(2)
                    )
                )
            );
        }
        return identToLagThreshold;
    }

    @Bean(initMethod = "afterPropertiesSet")
    public LogBrokerReaderService readerService(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LogBrokerClient logBrokerClient,
        @Autowired PartitionManager partitionManager,
        @Autowired MonitoringConfig monitoringConfig,
        @Autowired PartitionDao partitionDao,
        @Autowired ConfigurationService configurationService,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired LogShatterService logShatterService,
        @Autowired LogshatterMetaDao logshatterMetaDao,
        @Autowired ReadSemaphore readSemaphore,
        @Autowired BatchErrorLoggerFactory batchErrorLoggerFactory,
        @Value("${logshatter.logbroker.read-threads}") int readThreads,
        @Value("${logshatter.logbroker.max-messages-per-read}") int maxMessagesPerRead,
        @Value("${logshatter.logbroker.max-mb-per-read}") int maxMbPerRead

    ) {
        if (newApi) {
            return null;
        }
        LogBrokerReaderService readerService = new LogBrokerReaderService(
            logBrokerClient, partitionDao, partitionManager, monitoringConfig,
            logBrokerConfigurationService
        );
        //TODO сделать нормальную передачу параметров через конструктор
        readerService.setConfigurationService(configurationService);
        readerService.setLogShatterService(logShatterService);
        readerService.setLogshatterMetaDao(logshatterMetaDao);
        readerService.setReadSemaphore(readSemaphore);
        readerService.setErrorLoggerFactory(batchErrorLoggerFactory);
        readerService.setLogbrokerReaderThreads(readThreads);
        readerService.setMaxMessagesPerRead(maxMessagesPerRead);
        readerService.setMaxMbPerRead(maxMbPerRead);
        return readerService;
    }

    @Bean
    public LogBrokerConfigurationService logBrokerConfigurationService(
        @Autowired ConfigurationService configurationService,
        @Value("${logshatter.logbroker.disabled-sources}") String disabledSources
    ) {
        return new LogBrokerConfigurationService(configurationService.getConfigs(), disabledSources);
    }

    @Bean(initMethod = "startAsync")
    public LogBrokerCleanupService logBrokerCleanupService(
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired LogshatterMetaDao logshatterMetaDao,
        @Value("${logshatter.logbroker.sources-cleanup.run-interval-hours}") int runIntervalHours,
        @Value("${logshatter.logbroker.sources-cleanup.cleanup-after-days}") int cleanupAfterDays
    ) {
        return new LogBrokerCleanupService(
            logBrokerConfigurationService, logshatterMetaDao, runIntervalHours, cleanupAfterDays
        );
    }

    @Bean
    public LogBrokerLagLogger logBrokerLagLogger(@Autowired PartitionDao partitionDao) {
        return new LogBrokerLagLogger(partitionDao);
    }

    @Bean
    public LogBrokerLagMonitoring logBrokerLagMonitoring(
        @Autowired PartitionDao partitionDao,
        @Autowired MonitoringConfig monitoringConfig,
        @Autowired LogShatterMonitoring monitoring,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService
    ) {
        return new LogBrokerLagMonitoring(
            partitionDao, monitoringConfig, monitoring, logBrokerConfigurationService.getIdentToSources()
        );
    }

    @Bean
    public LogBrokerLagWriterOldApi logBrokerLagWriterOldApi(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired LogBrokerClient logBrokerClient,
        @Autowired PartitionDao partitionDao,
        @Autowired PartitionManager partitionManager,
        @Autowired LogBrokerConfigurationService logBrokerConfigurationService,
        @Autowired LogShatterMonitoring monitoring,
        @Value("${logshatter.logbroker.offsets-fetching.retry-count-until-crit:5}")
            int offsetsFetchingRetryCountUntilCrit,
        @Value("${logshatter.logbroker.offsets-fetching.sleep-between-retries-millis:5000}")
            int offsetsFetchingSleepBetweenRetriesMillis
    ) {
        if (newApi) {
            return null;
        }
        return new LogBrokerLagWriterOldApi(
            logBrokerClient,
            partitionDao,
            partitionManager,
            logBrokerConfigurationService,
            monitoring,
            offsetsFetchingRetryCountUntilCrit,
            offsetsFetchingSleepBetweenRetriesMillis
        );
    }

    @Bean
    public PartitionLeadersLogger partitionLeadersLogger(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired PartitionManager partitionManager
    ) {
        if (newApi) {
            return null;
        }
        return new PartitionLeadersLogger(partitionManager);
    }

    @Bean
    public PartitionLeadersDebugLogger partitionLeadersDebugLogger(
        @Value("${logshatter.logbroker.api.new:false}") boolean newApi,
        @Autowired PartitionManager partitionManager
    ) {
        if (newApi) {
            return null;
        }
        return new PartitionLeadersDebugLogger(partitionManager);
    }

    @Bean
    public LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring(
        @Autowired MonitoringConfig monitoringConfig,
        @Autowired LogShatterMonitoring monitoring
    ) {
        return new LogBrokerSourcesWithoutMetadataMonitoring(monitoringConfig, monitoring);
    }

    @Bean(destroyMethod = "shutdown")
    public ExecutorService executorServiceForScheduledTasks() {
        return Executors.newScheduledThreadPool(
            5,
            new ThreadFactoryBuilder().setNameFormat("scheduled-tasks-%d").build()
        );
    }
}
