package ru.yandex.market.logshatter.spring;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryNTimes;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 01/06/2017
 */
@Configuration
@Import({
    LogshatterSpringClickHouseConfig.class,
    LogshatterSpringMongoConfig.class,
    LogshatterSpringReadersConfig.class,
    LogshatterSpringJettyConfig.class,
    LogshatterSpringLogbrokerConfig.class
})
public class LogshatterSpringMainConfig {
    private static final Logger log = LogManager.getLogger();

    @Value("${logshatter.zookeeper.quorum}")
    private String zookeeperQuorum;

    @Value("${logshatter.zookeeper.timeout-seconds}")
    private int zookeeperTimeoutMillis;

    @Value("${logshatter.zookeeper.prefix}")
    private String zookeeperPrefix;

    @Autowired
    private LogShatterMonitoring monitoring;

    @Bean(initMethod = "start")
    public CuratorFramework curatorFramework() {
        MonitoringUnit zkConnectionMonitoringUnit = new MonitoringUnit("ZooKeeper");
        monitoring.getClusterCritical().addUnit(zkConnectionMonitoringUnit);

        CuratorFramework curatorFramework = CuratorFrameworkFactory.builder()
            .connectString(zookeeperQuorum)
            .retryPolicy(new RetryNTimes(Integer.MAX_VALUE, zookeeperTimeoutMillis))
            .connectionTimeoutMs(zookeeperTimeoutMillis)
            .sessionTimeoutMs(zookeeperTimeoutMillis)
            .namespace(zookeeperPrefix)
            .build();

        curatorFramework.getConnectionStateListenable().addListener((client, newState) -> {
            switch (newState) {
                case CONNECTED:
                case RECONNECTED:
                    log.info("Got connection to ZK. Status " + newState);
                    zkConnectionMonitoringUnit.ok();
                    break;
                case LOST:
                case SUSPENDED:
                    zkConnectionMonitoringUnit.critical("Lost connection to ZK");
                    log.error("Lost connection to ZK. Status " + newState);
                    break;
                default:
                    throw new IllegalStateException("Unknown connection state:" + newState);
            }
        });

        return curatorFramework;
    }
}
