package ru.yandex.market.logshatter.url;

import com.google.common.base.Strings;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicHeader;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.common.util.db.BulkUpdater;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 05/05/15
 */
public class PageMatcherImpl implements InitializingBean, Runnable, PageMatcher {

    private static final Logger log = LogManager.getLogger();
    private static final int BATCH_SIZE = 1000;

    private JdbcTemplate jdbcTemplate;
    private TransactionTemplate transactionTemplate;
    private LogShatterMonitoring monitoring;

    private final MultiMap<String, String> urlToHost = new MultiMap<>();
    private final ConcurrentMap<String, PageTree> pageTrees = new ConcurrentHashMap<>();
    private final MonitoringUnit pageMatcherMonitoringUnit = new MonitoringUnit("PageMatcher");
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();

    private HttpClient httpClient;
    private HttpClient mobileHttpClient;

    private String configFile;
    private int reloadIntervalMinutes = 60;
    private int timeoutSeconds = 60;
    private int maxConnections = 5;
    private int retryCount = 10;
    private int minutesSinceReloadToWarn;

    @Override
    public void afterPropertiesSet() throws Exception {
       if (Strings.isNullOrEmpty(configFile)) {
           log.info("Page matcher not configured");
           return;
        }
        pageMatcherMonitoringUnit.setWarningTimeout(reloadIntervalMinutes * 5, TimeUnit.MINUTES);
        monitoring.getHostCritical().addUnit(pageMatcherMonitoringUnit);
        readConfig();
        if (urlToHost.isEmpty()) {
            return;
        }
        createHttpClient();
        initDatabase();
        loadCache();
        scheduler.scheduleAtFixedRate(this, 0, reloadIntervalMinutes, TimeUnit.MINUTES);
        log.info("PageMatcher initialized");
    }

    private void initDatabase() {
        jdbcTemplate.update(
            "CREATE TABLE IF NOT EXISTS pages " +
                "(url VARCHAR(1000), page_id VARCHAR(1000), pattern VARCHAR(1000), TYPE VARCHAR(1000))"
        );
        jdbcTemplate.update("CREATE INDEX IF NOT EXISTS url_idx ON pages(url)");
        jdbcTemplate.update("ALTER TABLE pages ADD COLUMN IF NOT EXISTS reload_time TIMESTAMP DEFAULT '2000-01-01 00:00:00'");
    }

    private void loadCache() {
        log.info("Loading PageMatcher data from cache");
        for (Map.Entry<String, List<String>> entry : urlToHost.entrySet()) {
            String url = entry.getKey();
            List<Page> pages = loadFromCache(url);
            if (pages.isEmpty()) {
                log.info("No data in cache for url: " + url);
                return;
            }
            log.info("Found " + pages.size() + " pages in cache for url: " + url);
            PageTree pageTree = PageTree.build(pages);
            for (String host : entry.getValue()) {
                pageTrees.put(host, pageTree);
            }
        }
    }

    private List<Page> loadFromCache(String url) {
        return jdbcTemplate.query(
            "SELECT * FROM pages WHERE url = ?",
            new RowMapper<Page>() {
                @Override
                public Page mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return new Page(
                        rs.getString("page_id"),
                        rs.getString("pattern"),
                        rs.getString("type"),
                        rs.getTimestamp("reload_time").toInstant()
                    );
                }
            },
            url
        );
    }

    private void saveToCache(final String url, final List<Page> pages) {
        transactionTemplate.execute(new TransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus status) {
                jdbcTemplate.update("DELETE FROM pages WHERE url = ?", url);
                BulkUpdater bulkUpdater = new BulkUpdater(
                    jdbcTemplate,
                    "INSERT INTO pages (url, page_id, pattern, type, reload_time) values (?, ?, ?, ?, CURRENT_TIMESTAMP)",
                    BATCH_SIZE
                );
                for (Page page : pages) {
                    bulkUpdater.submit(url, page.getId(), page.getPattern(), page.getType());
                }
                bulkUpdater.done();
            }
        });
        log.info("PageMatcher data saved to cache for url: " + url);
    }

    @Override
    public void run() {
        try {
            reload();
        } catch (Exception e) {
            pageMatcherMonitoringUnit.warning("Reload failed", e);
            log.error("Reload failed", e);
        }
    }

    protected void readConfig() throws IOException {
        File file = new File(configFile);
        if (!file.isFile()) {
            log.error("Invalid config file for page matcher: " + configFile);
            pageMatcherMonitoringUnit.critical("No config for page matcher: " + configFile);
            return;
        }

        Properties config = new Properties();
        config.load(new FileInputStream(configFile));
        for (String host : config.stringPropertyNames()) {
            String url = config.getProperty(host);
            urlToHost.append(url.trim(), host.trim());
        }
        if (urlToHost.isEmpty()) {
            log.warn("Empty config for page matcher: " + configFile);
        }
    }

    protected void createHttpClient() {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(timeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        HttpClientBuilder builder = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .setRetryHandler(new DefaultHttpRequestRetryHandler(retryCount, true));

        httpClient = builder.build();
        mobileHttpClient = builder
            .setDefaultHeaders(Arrays.asList(
                new BasicHeader("User-Agent", "Mozilla/5.0 (iPhone; CPU iPhone OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9A334 Safari/7534.48.3")
            ))
            .build();
    }

    protected void reload() {
        if (urlToHost.isEmpty()) {
            return;
        }
        log.info("Reload started");
        List<String> failedUrls = new ArrayList<>();
        for (Map.Entry<String, List<String>> entry : urlToHost.entrySet()) {
            String url = entry.getKey();
            try {
                reload(url, entry.getValue());
            } catch (Exception e) {
                failedUrls.add(url);
                log.error("Failed to fetch pages from url: " + url, e);
            }
        }
        List<String> failedUrlsWithExpiredCache = failedUrls.stream()
            .filter(url -> getLastSuccessfulReloadInstantForUrl(url).isBefore(Instant.now().minus(minutesSinceReloadToWarn, ChronoUnit.MINUTES)))
            .collect(Collectors.toList());
        int reloaded = urlToHost.keyCount() - failedUrls.size();
        if (failedUrlsWithExpiredCache.isEmpty()) {
            pageMatcherMonitoringUnit.ok();
            log.info("Reload successfully complete. Reloaded {} urls. Failed to reload {} urls.", reloaded, failedUrls.size());
        } else {
            log.error("Reloaded {} urls. Failed to reload {} urls. Urls with expired cache: {}", reloaded, failedUrls.size(), failedUrlsWithExpiredCache);
            pageMatcherMonitoringUnit.warning(failedUrlsWithExpiredCache.size() + " urls with expired cache: " + failedUrlsWithExpiredCache);
        }
    }

    private Instant getLastSuccessfulReloadInstantForUrl(String url) {
        return loadFromCache(url).stream()
            .map(Page::getReloadInstant)
            .max(Instant::compareTo).orElse(Instant.MIN);
    }

    private void reload(String url, List<String> hosts) throws IOException {
        log.info("Fetching pages from url '" + url + "' for host(s): " + hosts);
        List<Page> pages = fetchPages(url);
        log.info("Fetched " + pages.size() + " from url: " + url);
        saveToCache(url, pages);
        PageTree pageTree = PageTree.build(pages);
        for (String host : hosts) {
            pageTrees.put(host, pageTree);
        }
    }

    private List<Page> fetchPages(String url) throws IOException {
        // TODO https://st.yandex-team.ru/MARKETHEALTH-408
        HttpClient client = url.contains("m.market.yandex.") ? mobileHttpClient : httpClient;
        HttpGet get = new HttpGet(url);
        try {
            HttpResponse response = client.execute(get);
            int httpCode = response.getStatusLine().getStatusCode();
            if (httpCode != HttpStatus.SC_OK) {
                String message = IOUtils.toString(response.getEntity().getContent());
                throw new IOException("Wrong http code:" + httpCode + ", message: " + message);
            }
            return Page.parsePages(response.getEntity().getContent());
        } finally {
            get.releaseConnection();
        }
    }

    @Override
    public Page matchUrl(String host, String httpMethod, String url) {
        if (host == null) {
            return null;
        }
        int portPosition = host.indexOf(':');
        if (portPosition > 0) {
            host = host.substring(0, portPosition);
        }

        PageTree pageTree = pageTrees.get(host);
        if (pageTree == null) {
            return null;
        }
        return pageTree.match(httpMethod, url);
    }

    @Required
    public void setJdbcTemplate(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    @Required
    public void setTransactionTemplate(TransactionTemplate transactionTemplate) {
        this.transactionTemplate = transactionTemplate;
    }

    @Required
    public void setMonitoring(LogShatterMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setConfigFile(String configFile) {
        this.configFile = configFile;
    }

    public void setTimeoutSeconds(int timeoutSeconds) {
        this.timeoutSeconds = timeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    public void setRetryCount(int retryCount) {
        this.retryCount = retryCount;
    }

    public void setReloadIntervalMinutes(int reloadIntervalMinutes) {
        this.reloadIntervalMinutes = reloadIntervalMinutes;
    }

    public void setMinutesSinceReloadToWarn(int minutesSinceReloadToWarn) {
        this.minutesSinceReloadToWarn = minutesSinceReloadToWarn;
    }
}
