package ru.yandex.market.logshatter.url;


import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.common.util.collections.MultiMap;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/04/15
 */
public class PageTree {

    private static final Logger log = LogManager.getLogger();

    private static final Map<Level.Type, Level> VARIABLES = Stream.of(Level.Type.METHOD, Level.Type.PATH)
        .collect(Collectors.toMap(v -> v, v -> new Level("<>", v)));
    private static final Map<Level, TreeLevel> EMPTY_MAP = Collections.emptyMap();

    private final TreeLevel root;

    public static PageTree build(InputStream inputStream) throws IOException {
        return build(Page.parsePages(inputStream));
    }

    public static PageTree build(List<Page> pages) {
        TreeLevel root = buildLevel(pages, 0);
        return new PageTree(root);
    }

    private static TreeLevel buildLevel(List<Page> pages, int levelNumber) {
        MultiMap<Level, Page> levelToPage = new MultiMap<>();
        Page currentLevelPage = null;
        for (Page page : pages) {
            if (page.hasLevel(levelNumber)) {
                Level level = page.getLevel(levelNumber);
                levelToPage.append(checkVariable(level), page);
            } else {
                if (currentLevelPage == null) {
                    currentLevelPage = page;
                } else {
                    log.warn("Two pages on the same level (using first):" + currentLevelPage + " and " + page);
                }
            }
        }
        if (levelToPage.isEmpty()) {
            return new TreeLevel(currentLevelPage, EMPTY_MAP);
        }
        Map<Level, TreeLevel> children = new HashMap<>();
        for (Map.Entry<Level, List<Page>> childEntry : levelToPage.entrySet()) {
            TreeLevel childLevel = buildLevel(childEntry.getValue(), levelNumber + 1);
            children.put(childEntry.getKey(), childLevel);
        }


        return new TreeLevel(currentLevelPage, children);
    }

    private static Level checkVariable(Level level) {
        String name = level.getName();
        if (VARIABLES.containsKey(level.getType()) && name.startsWith("<") && name.endsWith(">")) {
            return VARIABLES.get(level.getType());
        }
        return level;
    }

    public PageTree(TreeLevel root) {
        this.root = root;
    }

    public Page match(String method, String url) {
        List<Level> levels = Levels.parse(method, url);

        List<TreeLevel> currentLevel = Collections.singletonList(root);
        for (Level level : levels) {
            currentLevel = currentLevel.stream()
                .flatMap(current -> {
                    List<TreeLevel> children = current.getNext(level);
                    if (!children.isEmpty()) {
                        // если есть совпадаения, то берем их
                        return children.stream();
                    } else if (level.getType() == Level.Type.QUERY) {
                        // если нет совпадаений, но фильтруем по квери-параметру, берем параметр
                        return Stream.of(current);
                    } else {
                        // если нет совпадений и фильтурем не по квери-параметру, отфильтровываем
                        return Stream.empty();
                    }
                })
                .collect(Collectors.toList());
        }

        return currentLevel.stream().findFirst().map(TreeLevel::getPage).orElse(null);
    }

    private static class TreeLevel {
        private final Page page;
        private final Map<Level, TreeLevel> children;

        TreeLevel(Page page, Map<Level, TreeLevel> children) {
            this.page = page;
            this.children = Collections.unmodifiableMap(children);
        }

        public Page getPage() {
            return page;
        }

        public List<TreeLevel> getNext(Level level) {
            Optional<TreeLevel> next = Optional.ofNullable(children.get(level));

            if (VARIABLES.containsKey(level.getType())) {
                Optional<TreeLevel> variable = Optional.ofNullable(children.get(VARIABLES.get(level.getType())));
                return Stream.of(next, variable)
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .collect(Collectors.toList());
            } else {
                return next.map(Collections::singletonList)
                    .orElseGet(Collections::emptyList);
            }
        }
    }

}
