# coding: utf-8
import json

from direct.infra.mysql_manager.libs.helpers import read_file, check_json_schema_or_die, json_schema_make_all_required
from kazoo.hosts import collect_hosts


class DtMysqlFailoverManagerConfig:
    _schema = {
        "type": "object",
        "properties": {
            "status_dir": {"type": "string", "minLength": 1},
            "guard_iteration_sleep": {"type": "number"},
            "zk_prefix": {"type": "string", "minLength": 1},
            "kazoo_params": {
                # kazoo_params["timeout"] - это session_timeout zookeeper (время жизни эфемерных нод)
                # и он же connection timeout к нодам - отдельной настройки для этого не нашел =(
                # таймаут на первоначальное подключение отдельный, для реконнектов на все ноды его нужно ставить в
                # kazoo_params["timeout"] * ip_count (если хосты резолвятся более чем в 1 ip)
                "type": "object",
                "properties": {
                    "hosts": {"type": "string", "minLength": 1},
                    "timeout": {"type": "integer"},
                }
            },
            # специфичные для приложения параметры zk
            "zk_params": {
                "type": "object",
                "properties": {
                    "zk_token_file": {"type": "string", "minLength": 1},
                    "zk_start_timeout": {"type": "integer"},
                    "zk_read_modify_write_max_tries": {"type": "integer"},
                }
            },
            "mysql_params": {
                "type": "object",
                "properties": {
                    "timeout": {"type": "integer"},
                    "connect_timeout": {"type": "integer"},
                },
            },
            "mysql_command_max_tries": {"type": "integer"},
            "switchover_waiting_steps_timeout": {"type": "integer"},
            "switchover_waiting_iteration_sleep": {"type": "integer"},
            "switchover_min_trx_silence_time": {"type": "integer"},
            "switchover_max_seconds_behind_master": {"type": "integer"},
            "failover_pings_window": {"type": "integer"},
            "failover_pings_iteration_sleep": {"type": "integer"},
            "failover_min_failed_pings": {"type": "integer"},
            "failover_cooldown_time": {"type": "integer"},
            "ext_tools_timeout": {"type": "number"},
            "ext_tools": {
                "type": "object",
                "properties": {
                    "lm": {"type": "string", "minLength": 1},
                    "lfw": {"type": "string", "minLength": 1},
                    "sudo": {"type": "string", "minLength": 1},
                }
            }
        }
    }
    default_path = "/etc/dt-mymgr/config.json"
    description = "mysql failover manager config"

    def __init__(self, instance, config_file, read_passwords=True):
        self._conf = json.loads(read_file(config_file))

        json_schema_make_all_required(self._schema)
        check_json_schema_or_die(self._conf, self._schema)
        for attr in self._schema["properties"].keys():
            self.__setattr__(attr, self._conf[attr])

        self.instance = instance
        self.conf_path = config_file

        self.zk_path = "%s/%s" % (self.zk_prefix, self.instance)
        self.guard_zk_state_path = self.zk_path + "/guard-state"
        self.guard_zk_lock_path = self.zk_path + "/guard-lock"
        self.guard_zk_heartbeat_path = self.zk_path + "/guard-heartbeat"
        self.guard_status_file_path = f"{self.status_dir}/{self.instance}.zkguard-status"
        self.switchover_zk_lock_path = self.zk_path + "/switchover-lock"
        self.switchover_zk_state_path = self.zk_path + "/switchover-state"

        self.failover_zk_lock_path = self.zk_path + "/failover-lock"
        self.failover_zk_state_path = self.zk_path + "/failover-state"

        # вытаскиваем на верхний уровень для совместимости и удобства
        for attr in self.zk_params.keys():
            self.__setattr__(attr, self.zk_params[attr])
        # zk_token_file оставляем только на верхнем уровне, потому что zk_params с готовым токеном хочется передавать
        # напрямую в ZKClient
        self.zk_params.pop("zk_token_file", None)

        self.zk_hosts, chroot = collect_hosts(self.kazoo_params["hosts"])
        self.zk_ping_timeout = max(self.zk_start_timeout / len(self.zk_hosts), 1)
        assert all(isinstance(h, str) and isinstance(p, int) for h, p in self.zk_hosts) and chroot is None

        if read_passwords:
            self.read_passwords()

    def read_passwords(self):
        self.zk_params["zk_token"] = read_file(self.zk_token_file).strip()
        self.__setattr__("zk_token", self.zk_params["zk_token"])

    def get_zk_params(self):
        return dict(**self.zk_params, **self.kazoo_params)


class DtAllDBInstanceConfig:
    _schema = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "minLength": 1},
            "db_config": {"type": "string", "minLength": 1},
            "db_config_zk": {"type": "string", "minLength": 1},
            "db_config_master_node": {"type": "string", "minLength": 1},
            "db_config_port_node": {"type": "string", "minLength": 1},
            "lfw_port": {"type": "integer"},
            "mysql_port": {"type": "integer"},

            # mysql default readonly user
            "mysql_user": {"type": "string", "minLength": 1},
            "mysql_pass_file": {"type": "string", "minLength": 1},
            # mysql superuser
            "mysql_user_super": {"type": "string", "minLength": 1},
            "mysql_pass_file_super": {"type": "string", "minLength": 1},
            # пользователь для репликации
            "mysql_user_rplcat": {"type": "string", "minLength": 1},
            "mysql_pass_file_rplcat": {"type": "string", "minLength": 1},
            "auto_failover": {
                "type": "object",
                "properties": {
                    "enabled": {"type": "boolean"},
                    "min_alive_dc": {"type": "integer"}
                }
            },
            "semisync_enable": {"type": "boolean"},
            "replicas": {
                "type": "array",
                "items": {
                    "type": "object",
                    "properties": {
                        "host": {"type": "string", "minLength": 1},
                        "dc": {"enum": ["iva", "myt", "sas", "vla", "man"]},
                        "switchover_weight": {"type": "integer"},
                        "maintenance": {"type": "boolean"}
                    }
                },
                "minItems": 1
            }
        }
    }
    default_path = "/etc/yandex-direct/alldb-config.json"
    description = "direct alldb config"

    def __init__(self, instance, config_file, read_passwords=True):
        self._conf = json.loads(read_file(config_file))["instances"][instance]

        json_schema_make_all_required(self._schema)
        check_json_schema_or_die(self._conf, self._schema)
        for attr in self._schema["properties"].keys():
            self.__setattr__(attr, self._conf[attr])

        for attr in self._schema["properties"]["auto_failover"]["properties"].keys():
            self.__setattr__("auto_failover_" + attr, self._conf["auto_failover"][attr])

        self.instance = instance
        self.conf_path = config_file

        self.replicas_dict = {}
        for item in self.replicas:
            self.replicas_dict[item["host"]] = item

        if read_passwords:
            self.read_passwords()

    def get_switchover_weight(self, host):
        return self.replicas_dict[host]["switchover_weight"]

    def get_maintenance(self, host):
        return self.replicas_dict[host]["maintenance"]

    def get_dc(self, host):
        return self.replicas_dict[host]["dc"]

    def read_passwords(self):
        for key in ["mysql_pass_file", "mysql_pass_file_super", "mysql_pass_file_rplcat"]:
            if key in self._conf:
                self.__setattr__(key.replace("_file", ""), read_file(self._conf[key]).strip())
                self.__delattr__(key)

    @property
    def replica_hosts(self):
        return [r["host"] for r in self.replicas]

    @property
    def ready_replica_hosts(self):
        return [r["host"] for r in self.replicas if not r["maintenance"]]
