# coding: utf-8
import json
import logging

import kazoo
import kazoo.exceptions
import library.python.retry as retry_lib
from direct.infra.mysql_manager.libs.configs import DtAllDBInstanceConfig, DtMysqlFailoverManagerConfig
from direct.infra.mysql_manager.libs.helpers import set_json_value, get_json_value
from direct.infra.mysql_manager.libs.zookeeper import ZKClient, zk_jdumps

_logger = logging.getLogger(__name__)
_logger.addHandler(logging.NullHandler())


class DtDBConfigManager(ZKClient):
    def __init__(self, db_config_zk_path, db_config_type="direct", *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.db_config_zk_path = db_config_zk_path
        self.db_config_type = db_config_type

    def get_master(self, instance_config: DtAllDBInstanceConfig):
        db_config, zstat = self.zk.get(self.db_config_zk_path)
        db_config = json.loads(db_config)
        return get_json_value(db_config, instance_config.db_config_master_node)

    @retry_lib.retry_intrusive
    def set_master(self, instance_config: DtAllDBInstanceConfig, new_master):
        db_config, zstat = self.zk.get(self.db_config_zk_path)
        db_config = json.loads(db_config)
        set_json_value(db_config, instance_config.db_config_master_node, new_master)
        self.zk.set(self.db_config_zk_path, zk_jdumps(db_config, pretty=True), zstat.version)


class DtMysqlGuardManager(ZKClient):
    # тут нельзя делать None, т.к. "no_master" (нет мастера, никто не мастер сейчас)
    # и None (не знаю, кто мастер, я даже законнектиться к zk не могу) - это разные состояния zkguard!
    NO_MASTER = "no_master"
    DEFAULT_GUARD_ZK_STATE = {
        "master_fqdn": NO_MASTER,
        "zkguard_enabled": False,
    }

    def __init__(self, mymgr_config: DtMysqlFailoverManagerConfig, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.mymgr_conf = mymgr_config

    def init_guard_state(self, force=False):
        self.logger.info(f"init zkguard nodes for {self.mymgr_conf.instance}")

        self.zk.ensure_path(self.mymgr_conf.guard_zk_lock_path)
        try:
            self.zk.create(self.mymgr_conf.guard_zk_state_path, zk_jdumps(self.DEFAULT_GUARD_ZK_STATE), makepath=True)
            self.logger.info(f"new zkguard state: {self.DEFAULT_GUARD_ZK_STATE}")
        except kazoo.exceptions.NodeExistsError:
            self.logger.info(f"already initialized at {self.mymgr_conf.guard_zk_state_path}")

        if force:
            self.logger.info(f"forced set zkguard state for {self.mymgr_conf.instance}")
            self.zk.set(self.mymgr_conf.guard_zk_state_path, zk_jdumps(self.DEFAULT_GUARD_ZK_STATE))
            self.logger.info(f"new zkguard state: {self.DEFAULT_GUARD_ZK_STATE}")

    def get_master(self):
        return json.loads(self.zk.get(self.mymgr_conf.guard_zk_state_path)[0])["master_fqdn"]

    def get_state(self):
        return json.loads(self.zk.get(self.mymgr_conf.guard_zk_state_path)[0])

    @retry_lib.retry_intrusive
    def set_master(self, value):
        state, zstat = self.zk.get(self.mymgr_conf.guard_zk_state_path)
        state = json.loads(state)
        state["master_fqdn"] = value
        self.zk.set(self.mymgr_conf.guard_zk_state_path, zk_jdumps(state), zstat.version)

    def close_master_lfw(self):
        self.set_master(self.NO_MASTER)
