# coding: utf-8
import logging
import os
import socket
import time

from direct.infra.mysql_manager.libs.helpers import jdumps, DEFAULT_RETRY_CONF
from kazoo.client import KazooClient
import kazoo.security
import kazoo.exceptions


_logger = logging.getLogger(__name__)
_logger.addHandler(logging.NullHandler())


def zk_jdumps(data, **kwargs) -> bytes:
    params = dict(newline=True, safe_serialization=True)
    params.update(kwargs)
    return jdumps(data, **params).encode()


def zk_generate_lock_data(extra_data=None, **kwargs) -> str:
    data = {
        "timestamp": time.time(),
        "fqdn": socket.getfqdn(),
        "pid": os.getpid()
    }
    if extra_data:
        data.update(extra_data)
    return jdumps(data, newline=True, **kwargs)


def zk_make_default_acls(rw_token=None):
    acls = [
        kazoo.security.make_acl("ip", "127.0.0.1", all=True),
        kazoo.security.make_acl("world", "anyone", read=True),
    ]
    if rw_token:
        acls.append(kazoo.security.make_digest_acl(username=rw_token.split(":", 1)[0],
                                                   password=rw_token.split(":", 1)[1], all=True))
    return acls


def zk_init(zk_token=None, disable_kazoo_retries=False, **kazoo_params):
    zk_retry = {"max_tries": 10, "max_delay": 60, "ignore_expire": True}
    if disable_kazoo_retries:
        zk_retry["max_tries"] = 0
        zk_retry["ignore_expire"] = False

    for key in ["connection_retry", "command_retry"]:
        if key not in kazoo_params:
            kazoo_params[key] = zk_retry

    _logger.info(f"connecting to zk with params {kazoo_params}, {zk_retry}")
    if zk_token:  # токен, не файл с ним!
        # add_auth работает только после zk.start, при реконнектах теряется, поэтому передаем auth_data в конструктор
        if "auth_data" not in kazoo_params:
            kazoo_params["auth_data"] = set()
        kazoo_params["auth_data"].add(("digest", zk_token))

        if "default_acl" not in kazoo_params:
            kazoo_params["default_acl"] = zk_make_default_acls(zk_token)

    zk = KazooClient(**kazoo_params)
    return zk


class ZKClient:
    def __init__(self, zk_client=None, start_zk_now=False, zk_start_timeout=60, logger=None,
                 zk_read_modify_write_retry_conf=DEFAULT_RETRY_CONF, zk_read_modify_write_max_tries=None,
                 zk_token=None, disable_kazoo_retries=False, **kazoo_params):
        self.logger = _logger if not logger else logger
        self.logger = self.logger.getChild(self.__class__.__name__)
        self.zk_start_timeout = zk_start_timeout

        self.retry_conf = zk_read_modify_write_retry_conf
        if zk_read_modify_write_max_tries is not None:
            self.retry_conf = self.retry_conf.upto_retries(zk_read_modify_write_max_tries)

        if isinstance(zk_client, KazooClient):
            self.zk = zk_client
        else:
            self.zk = zk_init(zk_token=zk_token, disable_kazoo_retries=disable_kazoo_retries, **kazoo_params)
            if start_zk_now:
                self.zk_start()

    def zk_start(self):
        self.zk.start(timeout=self.zk_start_timeout)


def zk_check_acls(zk, node_path, required_acls):
    cur_acls, znode_stat = zk.get_acls(node_path)
    _logger.debug(f"current {node_path} acls: {cur_acls}")

    for acl in required_acls:
        if acl not in cur_acls:  # там namedtupl'ы, их можно так сравнивать
            _logger.debug(f"acl {acl} not exist in {node_path} acls {cur_acls}")
            return False

    _logger.debug(f"all required acls {required_acls} exist in {node_path} acls")
    return True
