#!/usr/bin/env python3
# coding: utf-8

import argparse
import json
import logging.handlers
import multiprocessing
import datetime

from direct.infra.mysql_manager.libs.helpers import init_logger
from direct.infra.mysql_manager.libs.configs import DtMysqlFailoverManagerConfig, DtAllDBInstanceConfig
from direct.infra.mysql_manager.libs.auto_failover import DtMysqlAutoFailover

LOG_FILE_PREFIX = "auto-failover"


def prepare_and_run_instance(instance, log_dir, mymgr_config_file, alldb_config_file):
    handler = logging.FileHandler(f"{log_dir}/{LOG_FILE_PREFIX}.{instance}.log"
                                  f".{datetime.datetime.now():%Y%m%d}")
    logger = init_logger(background_handler=handler)
    logger.info(f"running for instance {instance}")

    mymgr_conf = DtMysqlFailoverManagerConfig(instance, mymgr_config_file)
    inst_conf = DtAllDBInstanceConfig(instance, alldb_config_file)
    pinger = DtMysqlAutoFailover(mymgr_config=mymgr_conf, instance_config=inst_conf, logger=logger, start_zk_now=False,
                                 **mymgr_conf.get_zk_params())
    pinger.run()
    logger.info("finished")


def main():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument("-c", "--config", default=DtMysqlFailoverManagerConfig.default_path,
                        help=DtMysqlFailoverManagerConfig.description)
    parser.add_argument("-d", "--alldb-config", default=DtAllDBInstanceConfig.default_path,
                        help=DtAllDBInstanceConfig.description)
    parser.add_argument("-l", "--log-dir", required=True, help="директория для логов пошардовых переключателей")
    parser.add_argument("--apply", default=False, action="store_true")
    args = parser.parse_args()

    logger = init_logger()
    logger.info(f"running with args: {args}")

    with open(args.alldb_config) as f:
        alldb_config = json.load(f)

    failover_args = []
    for instance, inst_conf in alldb_config["instances"].items():
        # TODO: сделать возможность итерироваться по всем DtAllDBInstanceConfig в файле
        if inst_conf["project"] == "direct" and inst_conf["type"] == "mysql" and \
                inst_conf["auto_failover"]["enabled"]:
            failover_args.append((instance, args.log_dir, args.config, args.alldb_config))

    if not failover_args:
        logger.info("no instances with auto_failover_enabled in %s", args.alldb_config)
        return

    logger.info("going to run auto-failover processes for instances: %s", [x[0] for x in failover_args])

    if not args.apply:
        logger.info("dry-run mode, use --apply")
        return

    with multiprocessing.Pool(processes=len(failover_args)) as pool:
        pool.starmap(prepare_and_run_instance, failover_args)
    logger.info("finished")


if __name__ == "__main__":
    main()
