#!/usr/bin/env python3
# coding: utf-8
"""
Прототип демона для сбора статистики доступности реплик. Пока не понятно, нужен или нет.
"""

import argparse
import logging.handlers
import socket
import time

from direct.infra.mysql_manager.libs.configs import DtMysqlFailoverManagerConfig, DtAllDBInstanceConfig
from direct.infra.mysql_manager.libs.helpers import init_logger, tcp_ping

from direct.infra.mysql_manager.libs.zookeeper import ZKClient, zk_jdumps

_logger = logging.getLogger(__name__)
_logger.addHandler(logging.NullHandler())


class DtMysqlPinger(ZKClient):
    def __init__(self, mymgr_config: DtMysqlFailoverManagerConfig, instance_config: DtAllDBInstanceConfig,
                 *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.inst_conf = instance_config
        self.mymgr_conf = mymgr_config
        self.fqdn = socket.getfqdn()

        self.zk_start()

    def run(self):
        while True:
            for replica in self.inst_conf.ready_replica_hosts:
                self.logger.info(f"ping {replica}:{self.inst_conf.mysql_port} from {self.fqdn}")

                zk_path_suffix = f"/pinger/{self.fqdn}/{replica}/status_"
                result = {
                    "timestamp": time.time(),
                    # TODO: таймаут в конфиг
                    "mysql_available": tcp_ping(replica, self.inst_conf.mysql_port, timeout=3),
                }
                self.logger.info(f"ping result for {replica}:{self.inst_conf.mysql_port} from {self.fqdn}: {result}")
                self.zk.create(self.mymgr_conf.zk_path + zk_path_suffix, zk_jdumps(result), makepath=True,
                               sequence=True)
            time.sleep(self.mymgr_conf.failover_pings_iteration_sleep)


def main():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument("-c", "--config", default=DtMysqlFailoverManagerConfig.default_path,
                        help=DtMysqlFailoverManagerConfig.description)
    parser.add_argument("-d", "--alldb-config", default=DtAllDBInstanceConfig.default_path,
                        help=DtAllDBInstanceConfig.description)
    parser.add_argument("-i", "--instance", required=True, help="local mysql instance to guard")
    args = parser.parse_args()

    logger = init_logger()
    logger.info(f"running with args: {args}")

    mymgr_conf = DtMysqlFailoverManagerConfig(args.instance, args.config)
    inst_conf = DtAllDBInstanceConfig(args.instance, args.alldb_config)
    pinger = DtMysqlPinger(mymgr_config=mymgr_conf, instance_config=inst_conf, logger=logger, start_zk_now=False,
                           **mymgr_conf.get_zk_params())
    pinger.run()
    logger.info("finished")


if __name__ == "__main__":
    main()
