#!/usr/bin/env python3
# coding: utf-8
"""
"""

import argparse
import functools
import logging.handlers
import os

from direct.infra.mysql_manager.libs.configs import DtMysqlFailoverManagerConfig, DtAllDBInstanceConfig
from direct.infra.mysql_manager.libs.failover import DtMysqlFailoverManager
from direct.infra.mysql_manager.libs.auto_failover import DtMysqlAutoFailover
from direct.infra.mysql_manager.libs.helpers import init_logger, jdumps

# хотим видеть в интерактивном режиме сообщения только от заданного логгера и потомков
from direct.infra.mysql_manager.libs.mysql import mysql_cmd_on_result_ready, mysql_cmd_on_retry
from direct.infra.mysql_manager.libs.zk_managers import DtMysqlGuardManager, DtDBConfigManager

CONSOLE_LOGGER_NAME = "mysql_manager"
CONSOLE_LOGGER_LEVEL = logging.INFO
console_logger = logging.getLogger(CONSOLE_LOGGER_NAME)
console_logger.addHandler(logging.NullHandler())


def guard_init(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    mgr = DtMysqlGuardManager(mymgr_config=mymgr_conf, logger=console_logger, start_zk_now=True,
                              **mymgr_conf.get_zk_params())

    mgr.init_guard_state(force=args.force)


def guard_show(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    mgr = DtMysqlGuardManager(mymgr_config=mymgr_conf, logger=console_logger, start_zk_now=True,
                              **mymgr_conf.get_zk_params())
    print("mysql zkguard state:", jdumps(mgr.get_state(), pretty=False))


def switchover_state_init(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    mymgr = DtMysqlFailoverManager(instance_config=inst_conf, mymgr_config=mymgr_conf, logger=console_logger)
    mymgr.zk_client.zk_start()
    mymgr.init_switchover_state(force=args.force)


def switchover_state_show(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    mymgr = DtMysqlFailoverManager(instance_config=inst_conf, mymgr_config=mymgr_conf, logger=console_logger)
    mymgr.zk_client.zk_start()
    print("switchover state:\n" + jdumps(mymgr.get_switchover_state(), pretty=True))


def db_config_show(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    db_config = DtDBConfigManager(db_config_zk_path=inst_conf.db_config_zk, logger=console_logger,
                                  start_zk_now=True, **mymgr_conf.get_zk_params())
    print("current dbconfig master:", db_config.get_master(inst_conf))


def mymgr_states_init(instance, args):
    guard_init(instance, args)
    switchover_state_init(instance, args)
    failover_state_init(instance, args)


def mymgr_states_show(instance, args):
    db_config_show(instance, args)
    guard_show(instance, args)
    failover_state_show(instance, args)
    print()
    switchover_state_show(instance, args)


def failover_state_init(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    failover_mgr = DtMysqlAutoFailover(mymgr_config=mymgr_conf, instance_config=inst_conf, logger=console_logger,
                                       start_zk_now=True, **mymgr_conf.get_zk_params())
    failover_mgr.init_failover_state(force=args.force)


def failover_state_show(instance, args):
    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    failover_mgr = DtMysqlAutoFailover(mymgr_config=mymgr_conf, instance_config=inst_conf, logger=console_logger,
                                       start_zk_now=True, **mymgr_conf.get_zk_params())
    print("failover state:", jdumps(failover_mgr.get_failover_state(), pretty=False))


def switchover(instance, args):
    if not args.new_master:
        print("no --new-master specified, nothing to do (see --help)")
        return

    mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
    inst_conf = DtAllDBInstanceConfig(instance, args.alldb_config)
    indent_prefix = "    "

    # может и не надо пытаться сразу после получения результата что-то печатать,
    # понятный вывод можно делать и по-другому
    callbacks = {
        "mysql_command_on_result_ready": functools.partial(mysql_cmd_on_result_ready, prefix=indent_prefix,
                                                           logger=console_logger, level=CONSOLE_LOGGER_LEVEL),
        "mysql_command_on_retry": functools.partial(mysql_cmd_on_retry, prefix=indent_prefix,
                                                    logger=console_logger, level=CONSOLE_LOGGER_LEVEL),
    }
    under_maintenance = set(inst_conf.replica_hosts).difference(inst_conf.ready_replica_hosts)
    if under_maintenance:
        print("WARNING: next hosts will be excluded (maintenance mode enabled in alldb-config):", under_maintenance)

    print("starting failover manager", "in active mode" if args.apply else "in dry-run mode", "...")
    mymgr = DtMysqlFailoverManager(instance_config=inst_conf, mymgr_config=mymgr_conf, logger=console_logger,
                                   callbacks=callbacks)
    mymgr.zk_client.zk_start()
    mymgr.prepare(new_master=args.new_master,
                  old_master=args.old_master,
                  simple_switchover=not args.from_failed_master,
                  exclude_hosts=args.exclude_hosts)

    if not mymgr.switchover_precheck():
        if not args.ignore_prechecks:
            print("switchover prechecks failed, fix problems or exclude some hosts")
            return
        else:
            print("switchover prechecks failed, ignoring as requested")
    else:
        print("switchover prechecks passed")

    print("switchover plan:")
    for step in mymgr.switchover_steps:
        print(f"{indent_prefix}{step['index']}: {step['name']}")

    if not args.apply:
        print("dry-run mode, run with --apply")
        return

    if mymgr.switchover():
        print("\nswitched to:", mymgr.params.new_master)


def init_cmd_parsers():
    cmd_to_func = {
        "init": mymgr_states_init,
        "show": mymgr_states_show,
        "guard-init": guard_init,
        "guard-show": guard_show,
        "switchover-init": switchover_state_init,
        "switchover-show": switchover_state_show,
        "failover-init": failover_state_init,
        "failover-show": failover_state_show,
        "switchover": switchover,
    }

    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument("-c", "--config", default=DtMysqlFailoverManagerConfig.default_path,
                        help=DtMysqlFailoverManagerConfig.description)
    parser.add_argument("-d", "--alldb-config", default=DtAllDBInstanceConfig.default_path,
                        help=DtAllDBInstanceConfig.description)
    parser.add_argument("--force", default=False, action="store_true",
                        help="безусловно перезаписать стейт на дефолтное состояние")
    parser.add_argument("--apply", default=False, action="store_true",
                        help="на самом деле применять команду (не действует для *show команд)")
    parser.add_argument("--log-file", default=os.path.expanduser("~/mymgr.log"), help="куда писать подробный лог")
    parser.add_argument("--log-to-stdout", action="store_true", help="писать подробный лог на stdout")
    parser.add_argument("--disable-console-log", action="store_true", help="не писать человекочитаемый вывод")

    # switchover args - не хочу пока делать иерархическую cli
    parser.add_argument("-n", "--new-master", default=None,
                        help=f"fqdn нового мастера или '{DtMysqlFailoverManager.MASTER_AUTO}'"
                              "для автоматического выбора")
    parser.add_argument("-l", "--old-master", default=None, help="fqdn старого мастера (использовать осторожно)")
    parser.add_argument("--ignore-prechecks", action="store_true",
                        help="игнорировать предварительные проверки перед переключением (очень опасно)")
    parser.add_argument("--from-failed-master", default=False, action="store_true",
                        help="явно указывает, что старый мастер недоступен (влияет на способ переключения)")
    parser.add_argument("-x", "--exclude-host", dest="exclude_hosts", default=[], action="append",
                        help="исключить хост из обработки (можно указывать несколько раз)")

    parser.add_argument("instance", help="db instance name")
    parser.add_argument("cmd", choices=sorted(cmd_to_func.keys()), help="command to run")

    return parser, cmd_to_func


def main():
    parser, cmd_to_func = init_cmd_parsers()
    args = parser.parse_args()

    console_logger_name = CONSOLE_LOGGER_NAME
    background_log_handler = None
    if args.log_to_stdout:
        background_log_handler = "stdout"
    elif args.log_file:
        background_log_handler = logging.handlers.TimedRotatingFileHandler(args.log_file, when='D', backupCount=14)

    if args.disable_console_log:
        console_logger_name = None

    init_logger(console_logger_name=console_logger_name, console_level=CONSOLE_LOGGER_LEVEL,
                background_handler=background_log_handler)
    console_logger.debug(f"started with args: {args}")

    cmd_to_func[args.cmd](args.instance, args)


if __name__ == "__main__":
    main()
