#!/usr/bin/env python3
# coding: utf-8
"""
Генерирует runit-скрипты для найденных lm mysql инстансов
Удаляет сервисы для отсутствующих инстансов
"""

import argparse
import logging
import os
import stat
import shutil
from subprocess import check_output, check_call, STDOUT

from direct.infra.mysql_manager.libs.configs import DtMysqlFailoverManagerConfig

CONF_DIR = "/etc/sv"
SV_PREFIX = "dt-mysql-zkguard"
ZKGUARD_BINARY_NAME = "dt-mysql-zkguard"


def runit_install(conf, name):
    runit_conf_dir = CONF_DIR + "/" + name
    runit_conf_file = runit_conf_dir + "/run"

    os.makedirs(runit_conf_dir, exist_ok=True)
    with open(runit_conf_file, "w") as f:
        f.write(conf)
    st = os.stat(runit_conf_file)
    os.chmod(runit_conf_file, st.st_mode | stat.S_IEXEC)

    if not os.path.islink("/etc/init.d/" + name):
        os.symlink("/usr/bin/sv", "/etc/init.d/" + name)
    if not os.path.islink("/etc/service/" + name):
        os.symlink(runit_conf_dir, "/etc/service/" + name)

    logging.info("installed: " + runit_conf_file)


def runit_uninstall(name):
    check_call(["/usr/bin/sv", "force-stop", name])

    if os.path.islink("/etc/init.d/" + name):
        os.unlink("/etc/init.d/" + name)
    if os.path.islink("/etc/service/" + name):
        os.unlink("/etc/service/" + name)

    runit_conf_dir = CONF_DIR + "/" + name
    shutil.rmtree(runit_conf_dir, ignore_errors=True)
    logging.info("removed: " + runit_conf_dir)


def main():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument("-c", "--config", default=DtMysqlFailoverManagerConfig.default_path,
                        help=DtMysqlFailoverManagerConfig.description)
    parser.add_argument("--apply", default=False, action="store_true",
                        help="Записать новые и удалить лишние сервисы (иначе просто покажет, что собирается сделать)")
    parser.add_argument("action", choices=["install", "uninstall"], help="установить или удалить runit конфиги")
    parser.add_argument("instances", nargs="+",
                        help="список инстансов: all (все из lm или все установленные) или instance1 instance2 ...")
    args = parser.parse_args()
    logging.basicConfig(level=logging.DEBUG)

    if args.action == "install":
        if args.instances[0] == "all":
            lm_instances = check_output(["/usr/local/bin/lm", "--complete"], stderr=STDOUT).decode().split("\n")
            args.instances = [x for x in lm_instances if x.strip()]

        installed = 0
        for instance in args.instances:
            mymgr_conf = DtMysqlFailoverManagerConfig(instance, args.config)
            params = f"--config {mymgr_conf.conf_path} --instance {mymgr_conf.instance}"
            log = f"/var/log/mysql.{mymgr_conf.instance}/zkguard.log"
            sv_script = "\n".join((
                "#!/bin/bash -e",
                "sleep 1",
                "export PATH=/usr/local/sbin:/usr/local/bin:/sbin:/bin:/usr/sbin:/usr/bin",
                f"mkdir -p {mymgr_conf.status_dir}",
                f"chown monitor: -R {mymgr_conf.status_dir}",
                f"exec chpst -u monitor {ZKGUARD_BINARY_NAME} {params} >> {log} 2>&1",
            ))

            name = SV_PREFIX + "." + instance

            logging.info("going to install %s ..." % (name,))
            if args.apply:
                runit_install(sv_script, name)
                installed += 1
        logging.info("installed: %s" % (installed,))
    elif args.action == "uninstall":
        removed = 0
        for name in os.listdir(CONF_DIR):
            runit_conf_dir = CONF_DIR + "/" + name
            instance = name[len(SV_PREFIX) + 1:]
            if not os.path.isdir(runit_conf_dir) or not name.startswith(SV_PREFIX):
                continue
            if args.instances[0] != "all" and instance not in args.instances:
                continue

            logging.info("going to remove %s ..." % (name,))
            if args.apply:
                runit_uninstall(name)
                removed += 1
        logging.info("removed: %s" % (removed,))

    if not args.apply:
        logging.info("dry-run mode, use --apply")


if __name__ == "__main__":
    main()
