import logging
import requests

from ci.tasklet.common.proto import service_pb2 as ci
from direct.infra.newci.get_nanny_hosts.proto import schema_tasklet
from tasklet.services.yav.proto import yav_pb2 as yav

logger = logging.getLogger(__name__)

NANNY_HOST = 'http://nanny.yandex-team.ru'


class GetNannyHostsImpl(schema_tasklet.GetNannyHostsBase):
    def init_input_defaults(self):
        if not self.input.params.nanny_secret_name:
            self.input.params.nanny_secret_name = 'sec-01exf1rgke9yja9hy2h26sxpx7'
        if not self.input.params.nanny_secret_key:
            self.input.params.nanny_secret_key = 'nanny.token'

    def get_nanny_oauth_token(self):
        spec = yav.YavSecretSpec(
            uuid=self.input.params.nanny_secret_name, key=self.input.params.nanny_secret_key)
        nanny_oauth_token_value = self.ctx.yav.get_secret(spec).secret

        return nanny_oauth_token_value

    def get_nanny_service_hosts(self, service):
        logger.info("[Nanny]: Try to fetch nanny data for service: {}".format(service))

        nanny_oauth_token_value = self.get_nanny_oauth_token()
        endpoint = "/v2/services/{}/current_state/instances/partial/".format(service)

        response = requests.get(NANNY_HOST + endpoint,
                                headers={
                                    "Authorization": "OAuth {}".format(nanny_oauth_token_value),
                                    "Content-Type": "application/json",
                                    "Accept": "application/json"
                                })

        response_json = response.json()
        logger.info("[Nanny]: Fetched data: {}".format(response_json))

        nanny_hosts = [instances_part['container_hostname'] for instances_part in response_json['instancesPart']]
        logger.info("[Nanny]: Hosts: {}".format(nanny_hosts))

        return nanny_hosts

    def run(self):
        self.init_input_defaults()

        job_instance_id = self.input.context.job_instance_id
        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(job_instance_id)
        progress.id = "nannyHosts"
        progress.progress = 0
        nanny_services_count = len(self.input.params.nanny_services) or 1

        nanny_service_hosts = {}
        for nanny_service in self.input.params.nanny_services:
            nanny_service_hosts[nanny_service] = self.get_nanny_service_hosts(nanny_service)

            progress.progress += (1 / nanny_services_count)
            self.ctx.ci.UpdateProgress(progress)

        progress.progress = 1
        self.ctx.ci.UpdateProgress(progress)

        progress.status = ci.TaskletProgress.Status.SUCCESSFUL

        logger.info("[Nanny]: Hosts to put into output results: {}".format(nanny_service_hosts))

        for nanny_service in self.input.params.nanny_services:
            service_data = self.output.results.service_hosts.add()
            service_data.hosts[:] = nanny_service_hosts[nanny_service]
            service_data.name = nanny_service
