import os
import json
import logging

from ci.tasklet.common.proto import service_pb2 as ci
from direct.infra.newci.pci_dss_configs_check.proto import schema_tasklet
from sandbox.projects.common.vcs import arc
import sandbox.sdk2 as sandbox_sdk2

logger = logging.getLogger(__name__)

ARCADIA = 'ARCADIA'


def check_pci_dss_config_integrity(arc_root, config_path):
    logger.info("[PCI DSS]: verifying config {}".format(config_path))

    with open(os.path.join(arc_root, config_path), 'r') as file:
        config = json.loads(file.read())
        config_file_entries = config.get('data')

        for config_entry in config_file_entries:
            source_path = os.path.join(arc_root, config_entry.get('source').get('path'))
            source_type = config_entry.get('source').get('type')

            logger.info("[PCI DSS]: checking source path existence {}".format(source_path))

            if source_type == ARCADIA and not os.path.exists(source_path):
                logger.error("[PCI DSS]: {} do not exist".format(source_path))

                return False

    logger.info('[PCI DSS]: ok')

    return True


def test_pci_dss_configuration(arc_root, pci_dss_config_paths):
    for path in pci_dss_config_paths:
        check_result = check_pci_dss_config_integrity(arc_root, path)
        if not check_result:
            return False

    return True


class PciDssConfigsCheckImpl(schema_tasklet.PciDssConfigsCheckBase):
    def init_input_defaults(self):
        if not self.input.params.arc_secret_name:
            self.input.params.arc_secret_name = 'sec-01exf1rgke9yja9hy2h26sxpx7'
        if not self.input.params.arc_secret_key:
            self.input.params.arc_secret_key = 'arc.token'

    def run(self):
        self.init_input_defaults()

        arc_cli = arc.Arc(
            secret_name='{}[{}]'.format(self.input.params.arc_secret_name, self.input.params.arc_secret_key))

        logger.info("[PCI DSS]: Trying to fetch {} revision".format(self.input.context.target_commit.revision.hash))

        with arc_cli.mount_path("", self.input.context.target_commit.revision.hash, fetch_all=False) as arc_root:
            logger.info("[PCI DSS]: Mounted arc to {}".format(arc_root))

            check_result = test_pci_dss_configuration(arc_root, [config_path for config_path in
                                                                 self.input.params.pci_dss_config_paths])

            if check_result:
                job_instance_id = self.input.context.job_instance_id

                progress = ci.TaskletProgress()
                progress.job_instance_id.CopyFrom(job_instance_id)

                progress.id = "pciDssConfigsCheck"
                progress.progress = 1

                sandbox_task = sandbox_sdk2.Task.current
                sandbox_task_logs_resource_result = sandbox_sdk2.Resource.find(type='TASK_LOGS', task_id=sandbox_task.id)

                if sandbox_task_logs_resource_result:
                    sandbox_task_logs_resource = sandbox_task_logs_resource_result.first()

                    if sandbox_task_logs_resource is not None:
                        resource_id = sandbox_task_logs_resource.id
                        progress.module = "SANDBOX"
                        progress.url = "https://proxy.sandbox.yandex-team.ru/{}/common.log".format(str(resource_id))

                progress.status = ci.TaskletProgress.Status.SUCCESSFUL

                self.ctx.ci.UpdateProgress(progress)

                self.output.results.check_result = 'success'
            else:
                raise RuntimeError('[PCI DSS]: integrity check failed')
