import time
import random
import logging
import requests

import sandbox.sdk2 as sandbox_sdk2
from ci.tasklet.common.proto import service_pb2 as ci
from direct.infra.newci.pci_dss_review_check.proto import schema_tasklet
from tasklet.services.yav.proto import yav_pb2 as yav

logger = logging.getLogger(__name__)

ARCANUM_API_HOST = 'https://a.yandex-team.ru/api/v1'
ABC_API_HOST = 'https://abc-back.yandex-team.ru/api/v4'
STAFF_GAP_API_HOST = 'https://staff.yandex-team.ru/gap-api/api'

PCI_DSS_DOCS_URL = 'https://docs.yandex-team.ru/direct-dev/concepts/dev/pci-dss'

STATUS_SUCCESS = 'success'
STATUS_FAIL = 'fail'
STATUS_POLL_FOR_RESULT = 'poll'
STATUS_PCI_EXPRESS_EMPTY = 'pci_express_empty'

STATUS_BY_REVIEW_REQUEST_STATE = {
    'not_reviewed': STATUS_FAIL,
    'shipped': STATUS_SUCCESS,
    'approved': STATUS_SUCCESS,
    'fixed': STATUS_SUCCESS,
    'failed': STATUS_FAIL,
    'free': STATUS_SUCCESS,
    'unknown': STATUS_POLL_FOR_RESULT,
}

WAIT_TIMEOUT_FIVE_MIN = 5 * 60
WAIT_TIMEOUT_ITERATIONS_COUNT = 10

PCI_DSS_MESSAGE_PATTERN = "Требуется проверка в рамках PCI DSS"


def get_pci_dss_review_comment(certified_reviewer):
    return """{}. Автоматически назначен сертифицированный ревьюер @{}.
    Документация о процессе ревью в рамках стандарта PCI DSS: {}.
    При возникновении вопросов обращайтесь к @fomasha.
    """.format(PCI_DSS_MESSAGE_PATTERN, certified_reviewer, PCI_DSS_DOCS_URL)


def get_pci_express_legs_sandbox_tasks(pr_commit_id):
    pci_express_legs_sandbox_tasks = sandbox_sdk2.Task.find(type=['PCI_EXPRESS_LEGS'], description=str(pr_commit_id))

    logger.info("[PCI DSS]: no need to add comment. Comment exists: {}".format(pci_express_legs_sandbox_tasks))

    return pci_express_legs_sandbox_tasks


class PciDssReviewCheckImpl(schema_tasklet.PciDssReviewCheckBase):
    def init_input_defaults(self):
        if not self.input.params.arcanum_secret_name:
            self.input.params.arcanum_secret_name = 'sec-01f49pn04gd2xyk9xvwgnvmh38'
        if not self.input.params.arcanum_secret_key:
            self.input.params.arcanum_secret_key = 'arcanum-oauth-token'
        if not self.input.params.abc_service_name:
            self.input.params.abc_service_name = 'direct'
        if not self.input.params.abc_scope_name:
            self.input.params.abc_scope_name = 'development'
        if not self.input.params.abc_secret_name:
            self.input.params.abc_secret_name = 'sec-01f49pn04gd2xyk9xvwgnvmh38'
        if not self.input.params.abc_secret_key:
            self.input.params.abc_secret_key = 'abc-token'
        if not self.input.params.arc_secret_name:
            self.input.params.arc_secret_name = 'sec-01f49pn04gd2xyk9xvwgnvmh38'
        if not self.input.params.arc_secret_key:
            self.input.params.arc_secret_key = 'arc-token'
        if not self.input.params.staff_secret_name:
            self.input.params.staff_secret_name = 'sec-01f49pn04gd2xyk9xvwgnvmh38'
        if not self.input.params.staff_secret_key:
            self.input.params.staff_secret_key = 'staff-api-token'

    def get_arcanum_oauth_token(self):
        spec = yav.YavSecretSpec(
            uuid=self.input.params.arcanum_secret_name, key=self.input.params.arcanum_secret_key)
        arcanum_oauth_token_value = self.ctx.yav.get_secret(spec).secret

        return arcanum_oauth_token_value

    def get_abc_oauth_token(self):
        spec = yav.YavSecretSpec(
            uuid=self.input.params.abc_secret_name, key=self.input.params.abc_secret_key)
        abc_oauth_token_value = self.ctx.yav.get_secret(spec).secret

        return abc_oauth_token_value

    def get_staff_oauth_token(self):
        spec = yav.YavSecretSpec(
            uuid=self.input.params.staff_secret_name, key=self.input.params.staff_secret_key)
        staff_oauth_token_value = self.ctx.yav.get_secret(spec).secret

        return staff_oauth_token_value

    def get_pr_pci_dss_state(self):
        arcanum_oauth_token_value = self.get_arcanum_oauth_token()
        pr_number = self.input.context.launch_pull_request_info.pull_request.id
        pci_dss_review_state_query = 'fields=commit(id%2Cstate)%2Cfiles%2Ccomponents(name%2Cpackages)'
        pci_dss_endpoint = "/pci-dss/review-requests/{}?{}".format(
            pr_number, pci_dss_review_state_query)

        review_state_response = requests.get(ARCANUM_API_HOST + pci_dss_endpoint,
                                             headers={"Authorization": "OAuth {}".format(arcanum_oauth_token_value)})

        review_state_response_json = review_state_response.json()

        logger.info(
            "[PCI DSS]: Fetched review request {} state:".format(pr_number))
        logger.info(review_state_response_json)

        review_state = review_state_response_json['data']['commit']['state']
        review_commit_id = review_state_response_json['data']['commit']['id']

        logger.info(
            "[PCI DSS]: Current review request commit {} state is {}".format(review_commit_id, review_state))

        review_changed_packages = {
            package
            for item in review_state_response_json['data']['components']
            for package in item['packages']
        }

        logger.info("[PCI DSS]: changed packages {}".format(review_changed_packages))

        return review_commit_id, review_state, review_changed_packages

    def get_abc_service_members(self):
        if self.input.params.predefined_certified_reviewers:
            return [abc_service_member for abc_service_member in self.input.params.predefined_certified_reviewers]

        get_abc_oauth_token = self.get_abc_oauth_token()
        abc_members_query = "service__slug={}&role__scope={}".format(
            self.input.params.abc_service_name, self.input.params.abc_scope_name)
        if self.input.params.abc_role_code:
            abc_members_query += "&role__code={}".format(self.input.params.abc_role_code)

        abc_members_endpoint = "/services/members?{}".format(abc_members_query)

        abc_members_response = requests.get(ABC_API_HOST + abc_members_endpoint,
                                            headers={"Authorization": "OAuth {}".format(get_abc_oauth_token)})

        abc_members_response_json = abc_members_response.json()
        abc_members = [item['person']['login'] for item in abc_members_response_json['results']]

        logger.info("[PCI DSS]: Fetched abc members: {}", abc_members)

        return abc_members

    def get_staff_gap_info(self, login_list):
        staff_oauth_token_value = self.get_staff_oauth_token()

        staff_gap_response = requests.get(STAFF_GAP_API_HOST + '/availability',
                                          params={"l":  login_list, "include_holidays": 0},
                                          headers={"Authorization": "OAuth {}".format(staff_oauth_token_value)})

        staff_gap_response_json = staff_gap_response.json()

        logger.info("[PCI DSS]: Fetch staff gap data: {}".format(staff_gap_response_json))

        return staff_gap_response_json['persons']

    def get_arcanum_pr_comments(self):
        arcanum_oauth_token_value = self.get_arcanum_oauth_token()
        pr_number = self.input.context.launch_pull_request_info.pull_request.id
        review_request_comments_endpoint = "/review-requests/{}/comments".format(pr_number)

        review_request_comments_response = requests.get(ARCANUM_API_HOST + review_request_comments_endpoint,
                                                        headers={
                                                            "Authorization": "OAuth {}".format(
                                                                arcanum_oauth_token_value)})

        review_request_comments_response_json = review_request_comments_response.json()

        logger.info(
            "[PCI DSS]: Fetched review request {} comments:".format(pr_number))
        logger.info(review_request_comments_response_json)

        return [(comment['content'], comment['user']['name']) for comment in review_request_comments_response_json['data']]

    def get_arcanum_pr_reviewers(self):
        arcanum_oauth_token_value = self.get_arcanum_oauth_token()
        pr_number = self.input.context.launch_pull_request_info.pull_request.id
        review_request_info_endpoint = "/review-requests/{}?{}".format(
            pr_number, 'fields=assignees')

        review_request_info_response = requests.get(ARCANUM_API_HOST + review_request_info_endpoint,
                                                    headers={
                                                        "Authorization": "OAuth {}".format(arcanum_oauth_token_value)})

        review_request_info_response_json = review_request_info_response.json()

        logger.info(
            "[PCI DSS]: Fetched review request {} reviewers:".format(pr_number))
        logger.info(review_request_info_response_json)

        return [item['name'] for item in review_request_info_response_json['data']['assignees']]

    def patch_arcanum_pr_reviewers(self, next_reviewers):
        arcanum_oauth_token_value = self.get_arcanum_oauth_token()
        pr_number = self.input.context.launch_pull_request_info.pull_request.id
        review_request_change_info_endpoint = "/review-requests/{}/assignees".format(
            pr_number)
        review_request_change_info_body = {
            "assignees": next_reviewers
        }

        requests.put(ARCANUM_API_HOST + review_request_change_info_endpoint, json=review_request_change_info_body,
                     headers={"Authorization": "OAuth {}".format(arcanum_oauth_token_value)})

        logger.info("[PCI DSS]: Added reviewers: {}".format(next_reviewers))

    def post_arcanum_pr_comment(self, reviewer):
        arcanum_oauth_token_value = self.get_arcanum_oauth_token()
        pr_number = self.input.context.launch_pull_request_info.pull_request.id
        post_comment_query = 'fields=id,content,issue'
        post_comment_endpoint = "/pull-requests/{}/comments?{}".format(pr_number, post_comment_query)
        post_comment_body = {
            "content": get_pci_dss_review_comment(reviewer),
            "issue": False
        }

        requests.post(ARCANUM_API_HOST + post_comment_endpoint, json=post_comment_body,
                      headers={"Authorization": "OAuth {}".format(arcanum_oauth_token_value)})

        logger.info("[PCI DSS]: Posted comment to arcanum review")

    def get_certified_reviewer_to_assign(self, assigned_reviewers):
        commit_author = self.input.context.target_commit.author
        abc_service_certified_members = self.get_abc_service_members()
        staff_gap_info = self.get_staff_gap_info(abc_service_certified_members)

        available_to_assign_certified_reviewers = [reviewer
                                                   for reviewer in abc_service_certified_members
                                                   if reviewer != commit_author and
                                                   reviewer in staff_gap_info
                                                   and staff_gap_info[reviewer]['available_now']]
        assigned_certified_reviewers = set(assigned_reviewers) & set(available_to_assign_certified_reviewers)

        logger.info("[PCI DSS]: Assigned certified reviewers: {}".format(assigned_certified_reviewers))

        if len(assigned_certified_reviewers) > 0:
            return assigned_certified_reviewers.pop()

        logger.info("[PCI DSS]: Possible certified reviewers: {}".format(
            available_to_assign_certified_reviewers))

        random_reviewer = random.choice(available_to_assign_certified_reviewers)

        return random_reviewer

    def get_changed_target_packages(self, changed_packages):
        target_packages = set(self.input.params.pci_express_packages)
        changed_target_packages = target_packages & changed_packages

        logger.info("[PCI DSS]: target packages are {}".format(target_packages))
        logger.info("[PCI DSS]: changed target packages are {}".format(changed_target_packages))

        return changed_target_packages

    def process_pr_pci_dss_state_or_wait(self, iteration_num=0):
        pr_commit_id, pr_state, changed_packages = self.get_pr_pci_dss_state()
        pr_status = STATUS_BY_REVIEW_REQUEST_STATE.get(
            pr_state, STATUS_FAIL)

        if iteration_num == WAIT_TIMEOUT_ITERATIONS_COUNT:
            pci_express_legs_sandbox_tasks_query = get_pci_express_legs_sandbox_tasks(pr_commit_id)
            if not pci_express_legs_sandbox_tasks_query or pci_express_legs_sandbox_tasks_query.count == 0:
                logger.info("[PCI DSS]: No sandbox tasks found")
                pr_status = STATUS_PCI_EXPRESS_EMPTY

        if pr_status == STATUS_POLL_FOR_RESULT:
            logger.info("[PCI DSS]: wait for {} seconds".format(WAIT_TIMEOUT_FIVE_MIN))

            time.sleep(WAIT_TIMEOUT_FIVE_MIN)

            self.process_pr_pci_dss_state_or_wait(iteration_num + 1)
        else:
            job_instance_id = self.input.context.job_instance_id

            progress = ci.TaskletProgress()
            progress.job_instance_id.CopyFrom(job_instance_id)
            progress.id = "pciDssReviewCheck"
            progress.progress = 1
            progress.url = PCI_DSS_DOCS_URL

            with_changed_target_packages = len(self.get_changed_target_packages(changed_packages)) > 0

            if pr_status == STATUS_SUCCESS or pr_status == STATUS_PCI_EXPRESS_EMPTY or not with_changed_target_packages:
                progress.status = ci.TaskletProgress.Status.SUCCESSFUL

                self.ctx.ci.UpdateProgress(progress)
                self.output.results.review_state = 'success'
            else:
                assigned_reviewers = self.get_arcanum_pr_reviewers()
                certified_reviewer_to_assign = self.get_certified_reviewer_to_assign(assigned_reviewers)

                if certified_reviewer_to_assign not in assigned_reviewers:
                    self.patch_arcanum_pr_reviewers(assigned_reviewers + [certified_reviewer_to_assign])

                pr_comments = self.get_arcanum_pr_comments()
                filtered_comments = list(filter(lambda comment: comment[1] == 'robot-twilight' and PCI_DSS_MESSAGE_PATTERN in comment[0], pr_comments))

                if len(filtered_comments) == 0:
                    self.post_arcanum_pr_comment(certified_reviewer_to_assign)
                else:
                    logger.info("[PCI DSS]: no need to add comment. Comment exists: {}".format(filtered_comments))

                progress.status = ci.TaskletProgress.Status.FAILED

                self.ctx.ci.UpdateProgress(progress)
                self.output.results.review_state = 'failure'

                raise RuntimeError('[PCI DSS]: integrity check failed')

    def run(self):
        self.init_input_defaults()
        self.process_pr_pci_dss_state_or_wait()
