import subprocess
import shlex
import os
import json
import logging
import random
import re
import requests
from datetime import datetime

from tasklet.services.ci import get_ci_env
from ci.tasklet.common.proto import service_pb2 as ci
from startrek_client import Startrek
from direct.infra.newci.tracker.proto import schema_tasklet
from tasklet.services.yav.proto import yav_pb2 as yav
from sandbox.projects.common.vcs import arc

ABC_HOST = 'https://abc-back.yandex-team.ru/api/v4'
ABC_URL = ABC_HOST + '/duty/on_duty/'
ABC_DEFAULT_DUTY = 'robot-twilight'

logger = logging.getLogger(__name__)


def get_command_output(command, cwd=None):
    logger.info("Executing command '{}'".format(command))
    return subprocess.check_output(shlex.split(command), stderr=subprocess.PIPE, cwd=cwd)


def is_commit_affects_app(arc_root, commit_hash, dependencies):
    log_output = get_command_output(
        '{} tool arc log --json -n 1 --name-only {}'.format(
            os.path.join(arc_root, "ya"),
            commit_hash
        ),
        cwd=arc_root
    )
    logger.info("Commit info: {}".format(log_output))
    log = json.loads(log_output)

    changed_files = [name_entry['path'] for name_entry in log[0]['names']]

    for path in changed_files:
        for dep in dependencies:
            if path.startswith(dep) and (len(path) == len(dep) or path[len(dep)] == '/'):
                logger.info("Found that commit {} affects application in file {}".format(commit_hash, path))
                return True
    logger.info("Skip commit {} since it doesn't affect application".format(commit_hash))
    return False


class ChangelogImpl(schema_tasklet.ChangelogBase):
    LIMIT = 500
    STATE_PATTERN = '<{internal_state\\n%%((?:.|\n)*)%%}>$'

    def init_input_defaults(self):
        if not self.input.params.app_name:
            raise RuntimeError('app_name is not specified')
        if not self.input.params.arc_secret_name:
            self.input.params.arc_secret_name = 'sec-01exf1rgke9yja9hy2h26sxpx7'
        if not self.input.params.arc_secret_key:
            self.input.params.arc_secret_key = 'arc.token'
        if not self.input.params.startrek_secret_name:
            self.input.params.startrek_secret_name = 'sec-01crgb2w2czemp9gazbqkzgdz6'
        if not self.input.params.startrek_secret_key:
            self.input.params.startrek_secret_key = 'startrek_ppc'
        if not self.input.params.startrek_queue:
            self.input.params.startrek_queue = 'DIRECT'
        if not self.input.params.ticket_type:
            self.input.params.ticket_type = 'Release'
        # if not self.input.params.ticket_components:
        #    raise RuntimeError('ticket_components is not specified')
        if not self.input.params.wrap_changelog_in_comment:
            self.input.params.wrap_changelog_in_comment = False
        if not self.input.params.startrek_unique_prefix:
            self.input.params.startrek_unique_prefix = 'release'

    def run(self):
        self.init_input_defaults()

        # Получаем секрет для доступа к стартреку
        spec = yav.YavSecretSpec(uuid=self.input.params.startrek_secret_name, key=self.input.params.startrek_secret_key)
        token_value = self.ctx.yav.get_secret(spec).secret

        client = Startrek(useragent='Direct CI', token=token_value)

        # Получаем unique-ключ релизного тикета (уникальный в рамках мажорной версии приложения)
        unique = '{}_{}_{}'.format(
            self.input.params.startrek_unique_prefix,
            self.input.params.app_name,
            self.input.context.version_info.major
        )
        tickets = list(client.issues.get_all(filter='unique:' + unique))
        if tickets:
            ticket = tickets[0]
            state = self.load_state(ticket) or {"commits": []}
        else:
            state = {"commits": []}
        logger.info('Loaded state: {}'.format(json.dumps(state)))

        # Если задача тасклета -- только поменять статус тикета, то делаем это и выходим
        if self.input.params.change_status_only:
            if not ticket:
                raise RuntimeError('change_status_only is true, but ticket was not found')
            self.apply_transitions_to_ticket(ticket)
            logger.info('Finish due to change_status_only')
            self.output.changelog.content.extend([])
            self.output.changelog.issues.extend([])
            self.output.changelog.hashes.extend([])
            self.output.changelog.ticket = ticket.key
            return

        # Получаем список коммитов, привязанных к релизу, через ручку ci.GetCommits (постранично)
        commits_response = None
        timeline_commits = []
        while not commits_response or commits_response.HasField('next'):
            request = ci.GetCommitsRequest()
            request.flow_launch_id = self.input.context.job_instance_id.flow_launch_id
            request.limit = ChangelogImpl.LIMIT
            request.ci_env = get_ci_env(self.input.context)
            # Релизы могут накладываться друг на друга (пока один ждёт выкладки, второй собирается и тестируется),
            # поэтому сборка следующего релиза не должна включать в себя коммиты предыдущего
            # Если же предыдущий релиз непоправимо плохой и не будет захотфикшен,
            # то его надо отменить до сборки следующего, либо перенести недостающие изменения в тикет вручную
            request.type = ci.GetCommitsRequest.Type.FROM_PREVIOUS_ACTIVE
            if commits_response and commits_response.HasField('next'):
                request.offset.branch = commits_response.next.branch
                request.offset.number = commits_response.next.number

            commits_response = self.ctx.ci.GetCommits(request)
            logger.info('GetCommits() returned {} commits'.format(len(commits_response.timeline_commits)))
            timeline_commits.extend(commits_response.timeline_commits)
        logger.info('Total commits discovered: {}'.format(len(timeline_commits)))

        # Если коммиты пересекаются с уже сохранёнными в тикете, то мы собираем хотфикс к отменённому релизу
        # и в этом случае нам нужно получить только новые коммиты, а старые заново не добавлять
        all_commits = [str(timeline_commit.commit.revision.hash) for timeline_commit in timeline_commits]
        old_commits = state['commits']
        if set(all_commits).intersection(set(old_commits)):
            new_commits = set(all_commits).difference(set(old_commits))
            timeline_commits = filter(lambda x: x.commit.revision.hash in new_commits, timeline_commits)
            do_filtering = False
        else:
            # По-умолчанию прикрепляем задачи всех новые коммитов для минорной версии релиза, так как считаем что это хот-фикс
            # Однако добавляем фильтрацию, если до этого в релизе не было никаких коммитов
            # Такая ситуация возможна, если в каком-то из предыдущих кубиков в графе произошёл сбой и релизный тикет при предыдущих запусках графа не создался
            is_old_commits_empty = len(old_commits) == 0
            do_filtering = False if (self.input.context.version_info.minor and not is_old_commits_empty) else True
        # Вычисляем новое состояние, объединяя все полученные хеши коммитов. Потом это состояние будет записано в тикет
        state['commits'] = state['commits'] + [commit for commit in all_commits if commit not in set(old_commits)]
        # И обновляем хеш коммита, для которого запущена сборка
        state['arc_revision'] = self.input.context.target_commit.revision.hash

        # Оставляем только те коммиты, которые повлияли на собираемое приложение
        arc_cli = arc.Arc(
            secret_name='{}[{}]'.format(self.input.params.arc_secret_name, self.input.params.arc_secret_key))
        logger.info(
            "Created arc client. Trying to fetch {} revision".format(self.input.context.target_commit.revision.hash))
        mount_point = arc_cli.mount_path("", self.input.context.target_commit.revision.hash, fetch_all=False)
        with mount_point:
            arc_root = mount_point._work_path
            logger.info("Mounted arc to {}".format(arc_root))

            app_dir = self.input.context.config_info.dir
            dir_graph_output = get_command_output('%s dump dir-graph --plain --split %s' % (
                os.path.join(arc_root, "ya"),
                os.path.join(arc_root, app_dir)
            ))
            logger.info("Dir graph: {}".format(dir_graph_output))
            dir_graph = json.loads(dir_graph_output)

            # Получаем список директорий с префиксом 'direct', от которых зависит собираемое приложение
            common_path = 'direct'

            deps = []
            predefined_dependencies_list = list(self.input.params.predefined_dependencies_list)

            if predefined_dependencies_list and len(predefined_dependencies_list) > 0:
                deps = predefined_dependencies_list
                logger.info("Predefined dependencies: {}".format(deps))
            else:
                deps = [
                    dep
                    for dep in dir_graph
                    if dep.startswith(common_path) and (len(dep) == len(common_path) or dep[len(common_path)] == '/')
                ]
                logger.info("Dependencies in /direct: {}".format(deps))

            self.add_dependencies_graph_by_dir_for_unittests(deps, app_dir)
            if do_filtering:
                # Для каждого коммита проверяем, аффектит ли он собираемое приложение
                filtered_commits = [
                    timeline_commit
                    for timeline_commit in timeline_commits
                    if is_commit_affects_app(arc_root, timeline_commit.commit.revision.hash, deps)
                ]
            else:
                filtered_commits = timeline_commits

        changelog = []
        issues = []
        hashes = []
        for c in filtered_commits:
            changelog += [self._format(c.commit)]
            issues += c.commit.issues
            hashes += [c.commit.revision.hash]

        self.output.changelog.content.extend(changelog)
        self.output.changelog.issues.extend(issues)
        self.output.changelog.hashes.extend(hashes)

        version = self.input.context.version_info.major
        if self.input.context.version_info.minor:
            version += '.{}'.format(self.input.context.version_info.minor)
        logger.info("Version from CI context: {}".format(version))

        # Создаём тикет или дописываем комментарий к имеющемуся
        client = Startrek(useragent='Direct CI', token=token_value)
        escape_seq = '%%' if self.input.params.wrap_changelog_in_comment else ''
        # Для хотфиксов форматирование немного отличается
        if self.input.context.version_info.minor:
            formatted_changelog = 'В релиз включён новый код:\n{}\n{}\n{}'.format(
                escape_seq,
                "\n\n".join(changelog),
                escape_seq
            )
        else:
            formatted_changelog = '<{{Изменения\n{}\n{}\n{}\n}}>\n'.format(
                escape_seq,
                "\n\n".join(changelog),
                escape_seq
            )
        formatted_summary = 'релиз {}: сборка из NewCI v{} - выложить 1.{}-1'.format(
            self.input.params.app_name,
            version,
            # Здесь берём svn-ревизию, но она будет правильной только для сборки релиза
            # для хотфиксов number будет уже просто порядковым номером, поэтому при обновлении summary
            # мы распарсим версию из имеющегося тикета и подставим её в обновлённое название
            # (см код ниже)
            self.input.context.target_commit.revision.number
        )
        # Ставим исполнителем текущего дежурного
        assignee = self.input.context.flow_triggered_by
        logger.info("Current expect assignee: %s" % assignee)
        if assignee == "robot-ci":
            assignee = self.get_current_duty(token_value)
            logger.info("Change assignee to %s" % assignee)

        # Создаём или получаем имеющийся (если тикет с таким unique уже был создан)
        ticket = client.issues.create(
            queue=self.input.params.startrek_queue,
            summary=formatted_summary,
            assignee=assignee,
            type={'name': self.input.params.ticket_type},
            components=[component for component in self.input.params.ticket_components],
            # Указываем unique, чтобы для мажорной версии релиза не создавать больше одного тикета
            unique=unique,
            description=''
        )
        # Если релизный тикет с таким уже есть, то его description не должен быть пустым
        # в этом случае добавляем комментарий и обновляем название тикета
        if ticket.description != '':
            # нужно обновить название тикета, но при этом надо сохранить базовую версию релиза
            # TODO: можно сделать это без парсинга, сохраняя версию в отдельное поле в тикете
            pattern = r"релиз +(?:[a-z0-9\-]+): (?:.*?) NewCI v(\d+(?:\.\d+)?) - выложить +(1\.[0-9]+(?:\.[0-9]+)?-[0-9]+)(.*)"
            match = re.search(pattern, ticket.summary)
            if match:
                base_version = match.group(2)
                formatted_summary = 'релиз {}: сборка из NewCI v{} - выложить {}'.format(
                    self.input.params.app_name,
                    version,
                    base_version
                )
            else:
                raise RuntimeError('Unable to parse base version from ticket summary')

            # Все апдейты в апи Стартрека выполняем с ignore_version_change, чтобы не было конфликтов с другими инструментами и ручными изменениями
            ticket.comments.create(text=formatted_changelog)
            ticket.update(summary=formatted_summary, ignore_version_change=True)
            description = ticket.description
        else:
            description = formatted_changelog
        # И обновляем описание тикета: обновляем/добавляем ссылку на граф в CI и state
        description = self.update_description_with_ci_url(description, self.input.context.ci_url)
        description = self.update_description_with_state(description, state)
        ticket.update(description=description, ignore_version_change=True)

        self.output.changelog.ticket = ticket.key
        logger.info("Startrek ticket: {}".format(ticket.key))

        # Двигаем тикет в нужный статус
        self.apply_transitions_to_ticket(ticket)

        # Выводим результат кубика
        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)  # Id текущей выполняющейся задачи
        progress.id = "tracker"  # Идентификатор состояния
        progress.progress = 1  # 100%
        progress.module = "STARTREK"  # icon
        progress.url = "https://st.yandex-team.ru/{}".format(ticket.key)
        progress.status = ci.TaskletProgress.Status.SUCCESSFUL
        self.ctx.ci.UpdateProgress(progress)

    def apply_transitions_to_ticket(self, ticket):
        if self.input.params.ticket_transitions:
            next_transition = self.get_next_transition(ticket)
            while next_transition:
                logger.info('Executing transition "{}" to ticket'.format(next_transition))
                ticket.transitions[next_transition].execute()
                next_transition = self.get_next_transition(ticket)
            logger.info('No more suitable transitions found at current ticket status')

    def get_next_transition(self, ticket):
        if not self.input.params.ticket_transitions:
            return None
        logger.info('Ticket in status {} now. Available transitions: {}'.format(ticket.status,
                                                                                [transition.id for transition in
                                                                                 ticket.transitions]))
        last_transition_found = None
        for transition in self.input.params.ticket_transitions:
            for available_transition in ticket.transitions:
                if transition == available_transition.id:
                    last_transition_found = transition
        return last_transition_found

    def load_state(self, ticket):
        if not ticket.description:
            return None
        match = re.search(ChangelogImpl.STATE_PATTERN, ticket.description)
        if match:
            try:
                return json.loads(match.group(1))
            except json.JSONDecodeError as e:
                logger.error("Can't decode state from json")
                raise e
        else:
            return None

    def add_dependencies_graph_by_dir_for_unittests(self, dep_graph, app_dir):
        deps_wo_libs = [d for d in dep_graph if
                        d != 'direct/libs']  # в зависимости почему-то попадает целый direct/libs, пока просто убираем: https://st.yandex-team.ru/DIRECT-126151#5f3e2d33980f6c5d74a19092
        common_test_dirs = [app_dir, 'direct/common', 'direct/core', 'direct/libs', 'direct/libs-internal']
        for dir in common_test_dirs:
            target = [d for d in deps_wo_libs if d == dir or d.startswith(dir + '/')]
            if len(target) == 0:
                continue
            dir_with_dep = self.output.dump_graph.add()
            dir_with_dep.dir = dir
            dir_with_dep.dep_graph = ';'.join(target)

    def update_description_with_state(self, ticket_description, state):
        state_str = '<{{internal_state\n%%{}%%}}>'.format(json.dumps(state, sort_keys=True, indent=4))
        match = re.search(ChangelogImpl.STATE_PATTERN, ticket_description)
        if match:
            return re.sub(ChangelogImpl.STATE_PATTERN, state_str, ticket_description)
        else:
            return ticket_description + '\n' + state_str

    def update_description_with_ci_url(self, ticket_description, ci_url):
        ci_url_str = 'Релиз в CI: {}\n'.format(ci_url)
        pattern = 'Релиз в CI\\: [^\\n]+\\n'
        match = re.search(pattern, ticket_description)
        if match:
            return re.sub(pattern, ci_url_str, ticket_description)
        else:
            return ci_url_str + '\n' + ticket_description

    def _format(self, commit):
        revision = ''
        if commit.revision.hash:
            revision = f'((https://a.yandex-team.ru/arc_vcs/commit/{commit.revision.hash} {commit.revision.hash[:10]}))'
        else:
            revision = f'((https://a.yandex-team.ru/arc/commit/{commit.revision.number} {commit.revision.number}))'

        message = commit.message

        return f'{revision}: {message}'

    def get_current_duty(self, token_value):
        logger.info("Try get current duty")
        new_duty = None

        if self.input.params.app_name in ['java-web']:
            schedule_id = '2228'
        elif self.input.params.app_name in ['uac']:
            schedule_id = '30782'  # https://abc.yandex-team.ru/services/direct/duty2/30782
            new_duty = True
        else:
            schedule_id = '30726'  # https://abc.yandex-team.ru/services/direct-app-duty/duty2/30726
            new_duty = True

        if new_duty:
            current_date = datetime.today().strftime('%Y-%m-%d')
            params = {
                'schedule': schedule_id,
                'date_from': current_date,
                'date_to': current_date,
                'with_watcher': '1'
            }
            response = requests.get(ABC_HOST + '/duty/shifts/', params=params,
                                    headers={"Authorization": 'OAuth %s' % token_value})
            current_duty_resp = list(filter(lambda item: item.get('end') != current_date, response.json()['results']))
        else:
            response = requests.get(ABC_URL, params={'schedule': schedule_id},
                                    headers={"Authorization": 'OAuth %s' % token_value})
            current_duty_resp = response.json()

        current_duty = current_duty_resp[random.randint(0, len(current_duty_resp) - 1)]['person']['login'] \
            if len(current_duty_resp) > 0 else ABC_DEFAULT_DUTY

        logger.debug('ABC duty response: {}'.format(response.json()))

        return current_duty
