# -*- coding: utf8 -*-

import os
import sys
import json
import time
import requests
import re
from collections import defaultdict
from flask import render_template, request, jsonify, Blueprint

from startrek_client import Startrek
from juggler_sdk import JugglerApi, CheckFilter

from direct.infra.observatorium.lib.models.juggler import CompleteEvents, RawEvents, Downtimes, CompleteEvent
from direct.infra.observatorium.lib.tools.juggler import grafana_response, get_check_def, add_comment_to_ticket, create_ticket, DOWNTIME_TICKET_COMPONENT
from direct.infra.observatorium.lib.tools.settings import DIRECT_PPC_STARTREK_TOKEN, JUGGLER_TOKEN

juggler_blueprint = Blueprint('juggler', __name__, url_prefix='/juggler')

juggler_api = JugglerApi("http://juggler-api.search.yandex.net:8998/api")

REMOVE_DOWNTIME_URL = 'http://juggler-api.search.yandex.net/v2/downtimes/remove_downtimes'
MAKE_DOWNTIME_URL = 'http://juggler-api.search.yandex.net/v2/downtimes/set_downtimes'
USER_AGENT = 'Direct.Observatorium'
DIRECT_NAMESPACE = 'direct.prod'
DAYS_FOR_DOWNTIME = 1
MAX_DAYS_FOR_DOWNTIME = '30'

try:
    from library.python import resource
    hosts = json.loads(resource.find("/data/hosts.json"))
except ImportError:
    with open(os.path.join(os.path.dirname(__file__), '..', 'data', 'hosts.json')) as fh:
        hosts = json.load(fh)

host_tags = {}
"""
делаем такое:
{
    'direct' : {
        'api.direct.yandex.ru': {
            'direct-admin': True,
            'important': True
        },
        'direct.db.ppcdata': {
            'important': True
        },
    }
}
"""
for project in hosts:
    if project not in host_tags:
        host_tags[project] = {}
    for h in hosts[project]:
        if h['host'] not in host_tags[project]:
            host_tags[project][h['host']] = defaultdict(str)
        for t in h['tags']:
            host_tags[project][h['host']][t] = True

"""
делаем такое:
{
    'direct.yandex.ru': 'direct',
    'bm.db.smth': 'bm',
}
"""
host_to_project_dict = {}
for project in hosts:
    for h in hosts[project]:
        host_to_project_dict[h['host']] = project


def host_to_project(host):
    if host in host_to_project_dict:
        return host_to_project_dict[host]
    else:
        return '-'


def count_host(ls):
    if not ls:
        return []
    new_ls = list(ls)
    new_ls.sort(key=lambda x: x['quantity'], reverse=True)
    return new_ls


def cmp_check_tag(x, y):
    if y['quantity'] == 0:
        return -1
    elif x['quantity'] == 0:
        return 1
    elif x['name'][:6] == 'a_mark' and y['name'][:6] == 'a_mark' \
            or x['name'][:6] != 'a_mark' and y['name'][:6] != 'a_mark':
        return y['quantity'] - x['quantity']
    elif y['name'][:6] == 'a_mark':
        return -1
    else:
        return 1


def count_check(ls):
    if not ls:
        return []
    new_ls = list(ls)
    new_ls = sorted(new_ls, cmp=cmp_check_tag)
    return new_ls


@juggler_blueprint.route('/')
@juggler_blueprint.route('/index')
def index():
    return render_template('juggler/index.html')


@juggler_blueprint.route('/ce/')
@juggler_blueprint.route('/ce/<path:params>')
def complete_events(params=''):
    form = {
        'project': request.args.get('project', ''),
        'host_mode': request.args.get('host_mode', 'plus'),
        'host': request.args.get('host', ''),
        'service_mode': request.args.get('service_mode', 'plus'),
        'service': request.args.get('service', ''),
        'host_tag': request.args.get('host_tag', ''),
        'check_tag': request.args.get('check_tag', ''),
        'include_status': {
            'CRIT': request.args.get('include_CRIT', False),
            'WARN': request.args.get('include_WARN', False),
            'OK': request.args.get('include_OK', False),
            },
        'format': request.args.get('format', 'html'),
        'need_children': request.args.get('need_children', 0),
        'autorefresh': request.args.get('autorefresh', False),
        'directadmintv': request.args.get('directadmintv', False),
    }

    if form['format'] == '':
        form['format'] = 'html'

    host_mode = form['host_mode']
    service_mode = form['service_mode']
    error_message = ''

    if not form['project']:
        return render_template(
            'juggler/complete_events.html',
            form=form,
            error_message=u'Пустые параметры поиска, показать нечего'
        )

    check_tags = []
    host_tags_to_filter = []

    hosts_to_search = [h['host'] for h in hosts[form['project']]]
    form_host = ''.join(form['host'])
    form_host = form_host.replace(u';', u',')
    form_host = form_host.replace(u' ', u',')
    hosts_to_filter = [x for x in form_host.split(',') if x != '']

    if len(hosts_to_filter) == 0:
        host_mode = 'plus'

    form_service = ''.join(form['service'])
    form_service = form_service.replace(u';', u',')
    form_service = form_service.replace(u' ', u',')
    services_to_filter = [x for x in form_service.split(',') if x != '']

    if len(services_to_filter) == 0:
        service_mode = 'plus'

    try:
        statuses = [st for st in form['include_status'] if form['include_status'][st]]
        if form['directadmintv']:
            events = CompleteEvents(status=statuses, apiv2=True, tags=[["directadmin_TV"]])
        else:
            events = CompleteEvents(hosts=hosts_to_search, status=statuses, need_children=form['need_children'])

        dict_check_tags = {}
        dict_host_tags = {}

        for event in events.get():
            for tag in event.tags:
                dict_check_tags[tag] = 0

        for h in hosts[form['project']]:
            for tag in h['tags']:
                dict_host_tags[tag] = 0

        events.filter_list_by_params(
            hosts={host: '' for host in hosts_to_filter},
            host_mode=host_mode,
            services={service: '' for service in services_to_filter},
            service_mode=service_mode,
            host_tags_filters=[host_tag for host_tag in form['host_tag'].split(',') if host_tag != ''],
            host_tags=host_tags,
            check_tags=[check_tag for check_tag in form['check_tag'].split(',') if check_tag != ''],
            form=form,
        )

        for event in events.get():
            for tag in event.tags:
                dict_check_tags[tag] += 1

        for event in events.get():
            for tag in host_tags[form['project']].get(event.host, []):
                if host_tags[form['project']][event.host][tag]:
                    dict_host_tags[tag] += 1

        check_tags = []
        for x in dict_check_tags:
            check_tags.append({'name': x, 'quantity': dict_check_tags[x]})

        host_tags_to_filter = []
        for x in dict_host_tags:
            host_tags_to_filter.append({'name': x, 'quantity': dict_host_tags[x]})

        check_tags = count_check(check_tags)
        host_tags_to_filter = count_host(host_tags_to_filter)

        # try:
        #     events.get_hints(project='direct')
        # except:
        #     pass

        events.find_ticket_for_downtimes()
        events.sort()

    except Exception:
        events = CompleteEvents(hosts=[], from_list=[])
        error_message = "Error getting events: %s" % [str(i) for i in sys.exc_info()]

    if form['format'] == 'html':
        return render_template(
            'juggler/complete_events.html',
            form=form,
            error_message=error_message,
            events=events.get(),
            check_tags=json.dumps(check_tags),
            host_tags=json.dumps(host_tags_to_filter),
        )
    elif form['format'] == 'json':
        json_events = {}
        for event in events.get():
            json_events.setdefault(event.host, []).append({
                event.name: {
                    'status': [event.status, event.status_begin_at],
                    'check_time': event.datetime,
                    'tags': [tag for tag in event.tags],
                    'summary': event.summary,
                    'description': event.description,
                }
            })

        return jsonify(json_events)

    elif form['format'] == 'graphite':
        statuses = {}
        statuses['CRIT'] = 0
        statuses['WARN'] = 0
        statuses['OK'] = 0
        statuses['downtimes'] = 0
        for event in events.get():
            statuses[event.status] += 1
            if re.search(u'Forced OK by downtimes', event.description, re.I):
                statuses['downtimes'] += 1

        statuses['total'] = statuses['CRIT'] + statuses['WARN'] + statuses['OK']

        return jsonify(statuses)


@juggler_blueprint.route('/re/')
@juggler_blueprint.route('/re/<path:params>')
def raw_events(params=''):
    form = {
        'project': request.args.get('project', ''),
        'host_mode': request.args.get('host_mode', 'plus'),
        'host': request.args.get('host', ''),
        'service_mode': request.args.get('service_mode', 'plus'),
        'service': request.args.get('service', ''),
        'host_tag': request.args.get('host_tag', ''),
        'include_status': {
            'CRIT': request.args.get('include_CRIT', False),
            'WARN': request.args.get('include_WARN', False),
            'OK': request.args.get('include_OK', False),
        },
        'format': request.args.get('format', 'html'),
    }

    if form['format'] == '':
        form['format'] = 'html'

    host_mode = form['host_mode']
    service_mode = form['service_mode']
    error_message = ''

    if not form['project']:
        return render_template(
            'juggler/raw_events.html',
            form=form,
            error_message=u'Пустые параметры поиска, показать нечего'
        )

    host_tags_to_filter = []

    hosts_to_search = [h['host'] for h in hosts[form['project']]]
    form_host = ''.join(form['host'])
    form_host = form_host.replace(u';', u',')
    form_host = form_host.replace(u' ', u',')
    hosts_to_filter = [x for x in form_host.split(',') if x != '']

    if len(hosts_to_filter) == 0:
        host_mode = 'plus'

    form_service = ''.join(form['service'])
    form_service = form_service.replace(u';', u',')
    form_service = form_service.replace(u' ', u',')
    services_to_filter = [x for x in form_service.split(',') if x != '']

    if len(services_to_filter) == 0:
        service_mode = 'plus'

    try:
        statuses = [st for st in form['include_status'] if form['include_status'][st]]
        events = RawEvents(hosts=hosts_to_search, status=statuses)

        dict_host_tags = {}

        for h in hosts[form['project']]:
            for tag in h['tags']:
                dict_host_tags[tag] = 0

        events.sort()
        events.filter_list_by_params(
            hosts={host: '' for host in hosts_to_filter},
            host_mode=host_mode,
            services={service: '' for service in services_to_filter},
            service_mode=service_mode,
            host_tags_filters=[host_tag for host_tag in form['host_tag'].split(',') if host_tag != ''],
            host_tags=host_tags,
            form=form,
        )

        for event in events.get():
            for tag in host_tags[form['project']][event.host]:
                if host_tags[form['project']][event.host][tag]:
                    dict_host_tags[tag] += 1

        host_tags_to_filter = []
        for x in dict_host_tags:
            host_tags_to_filter.append({'name': x, 'quantity': dict_host_tags[x]})

        host_tags_to_filter = count_host(host_tags_to_filter)

        # try:
        #     events.get_hints(project='direct')
        # except:
        #     pass

    except:
        events = RawEvents([])
        error_message = "Error getting events: %s" % [str(i) for i in sys.exc_info()]

    if form['format'] == 'html':
        return render_template(
            'juggler/raw_events.html',
            form=form,
            error_message=error_message,
            events=events.get(),
            host_tags=json.dumps(host_tags_to_filter),
        )
    elif form['format'] == 'json':
        json_events = {}
        for event in events.get():
            json_events.setdefault(event.host, []).append({
                event.name: {
                    'status': [event.status, event.time],
                    'check_time': event.datetime,
                    'description': event.description,
                }
            })

        return jsonify(json_events)


@juggler_blueprint.route('/checks/')
@juggler_blueprint.route('/checks/<path:params>')
def checks(params=''):
    form = {
        'namespace': request.args.get('namespace', DIRECT_NAMESPACE),
        'host_tag': request.args.get('host_tag', ''),
        'host': request.args.get('host', ''),
        'service': request.args.get('service', ''),
        'show_status': request.args.get('show_status', ''),
    }

    if form['namespace'] == '':
        form['namespace'] = DIRECT_NAMESPACE

    all_checks = juggler_api.get_checks([
        CheckFilter(namespace=form['namespace'], host=form['host'], service=form['service'])
    ])

    checks = [{
        'namespace': check.namespace,
        'host': check.host,
        'service': check.service,
        'static_props': {},
        'methods': check.alert_methods or [],
        'tags': check.tags or [],
    } for check in all_checks.checks]

    return render_template(
        'juggler/checks.html',
        form=form,
        checks=checks,
        error_message=u''
    )


@juggler_blueprint.route('/dt/')
@juggler_blueprint.route('/dt/<path:params>')
def downtimes(params=''):
    form = {
        'host': request.args.get('host', ''),
        'service': request.args.get('service', ''),
        'include_status': {
            'CRIT': request.args.get('include_CRIT', False),
            'WARN': request.args.get('include_WARN', False),
            'OK': request.args.get('include_OK', False),
        },
        'namespace': request.args.get('namespace', "all"),
    }

    if form['namespace'] == 'all':
        namespaces = ['direct.prod', 'direct.test', 'directmod.prod', 'bm.prod', 'partner.prod', 'advq.prod']
    else:
        namespaces = [form['namespace']]
    error_message = ''

    form_service = ''.join(form['service'])
    form_service = form_service.replace(u';', u',')
    form_service = form_service.replace(u' ', u',')
    services_to_filter = [x for x in form_service.split(',') if x != '']

    form_host = ''.join(form['host'])
    form_host = form_host.replace(u';', u',')
    form_host = form_host.replace(u' ', u',')
    hosts_to_filter = [x for x in form_host.split(',') if x != '']

    downtimes = Downtimes(hosts=hosts_to_filter, service_name=services_to_filter, namespace=namespaces)
    downtimes.sort()

    return render_template(
        'juggler/downtimes.html',
        form=form,
        error_message=error_message,
        events=downtimes.get(),
    )


def find_issue(ticket_name):
    client = Startrek(useragent=u'direct-tag-cloud',
                      token=DIRECT_PPC_STARTREK_TOKEN)
    issue = client.issues[ticket_name]
    return issue


@juggler_blueprint.route('/make_downtime/')
@juggler_blueprint.route('/make_downtime/<path:params>')
def make_downtime(params=''):
    form = {
        'object_name': request.args.get('object_name', ''),
        'service_name': request.args.get('service_name', ''),
        'ticket_name': request.args.get('ticket_name', ''),
        'ticket_summary': request.args.get('ticket_summary', '').strip(),
        'days_to_rm': request.args.get('days_to_rm', MAX_DAYS_FOR_DOWNTIME),
        'queue': request.args.get('queue', ''),
        'mode': request.args.get('mode', '0')
    }

    events_for_downtime = [
        {'host': host, 'service': service}
        for host, service in zip(form['object_name'].split(','), form['service_name'].split(','))
    ]
    current_events = CompleteEvents(apiv2=True, filters={'filters': events_for_downtime})

    if len(current_events.get()) == 0:
        current_events = RawEvents(
            hosts=[event['host'] for event in events_for_downtime],
            service_name=[event['service'] for event in events_for_downtime],
            with_obsolete=True
        )

    if len(current_events.get()) == 0:
        return jsonify({'objects': [{'success': False, 'message': 'Error: event not found'}]})

    if form['mode'] == '1' or form['mode'] == '2' and isinstance(current_events, CompleteEvents):
        no_crit_raw_events = True

        for event in current_events.get():
            event.find_crit_raw_events(mode=form['mode'])
            if len(event.crit_raw_events) > 0:
                no_crit_raw_events = False

        if no_crit_raw_events:
            return jsonify({'objects': [{'success': False, 'message': 'Error: critical raw events were not found'}]})

    try:
        if form['queue'] == '':
            issue = find_issue(form['ticket_name'])
            ticket_name = form['ticket_name'].upper()
            issue.update(components=issue.components + [DOWNTIME_TICKET_COMPONENT])
            if form['mode'] == '0':
                add_comment_to_ticket(
                    issue.key,
                    u'Поставлен даунтайм на проверку %s' % (
                        ', '.join(["%s: %s" % (event.host, event.name) for event in current_events.get()])
                    ),
                    u"make_downtime"
                )
            elif form['mode'] == '1':
                add_comment_to_ticket(
                    issue.key,
                    u'Поставлен даунтайм на сырые события для проверки %s:\n%s' % (
                        ', '.join(["%s: %s" % (event['host'], event['service']) for event in events_for_downtime]),
                        '\n'.join([
                            "%s" % crit_event['host']
                            for event in current_events.get() for crit_event in event.crit_raw_events
                        ])
                    ),
                    u"make_downtime"
                )
            elif form['mode'] == '2':
                add_comment_to_ticket(
                    issue.key,
                    u'Поставлен даунтайм на сырые события для проверки %s:\n%s' % (
                        ', '.join(["%s: %s" % (event['host'], event['service']) for event in events_for_downtime]),
                        '\n'.join([
                            "%s: %s" % (crit_event['host'], crit_event['service'])
                            for event in current_events.get() for crit_event in event.crit_raw_events
                        ])
                    ),
                    u"make_downtime"
                )
        else:
            issue = create_ticket(
                current_events.get(), form, u"make_downtime", isinstance(current_events, CompleteEvents)
            )
            ticket_name = issue.key

    except Exception:
        return jsonify({'objects': [{'success': False, 'message': 'Error: process with ticket failed'}]})

    req_data = {
        "start_time": int(time.time()),
        "end_time": int(time.time()) + DAYS_FOR_DOWNTIME * 24 * 60 * 60,
        "description": "startrek:%s\ndays_to_rm:%s" % (ticket_name, form['days_to_rm'])
    }

    if form['mode'] == '1' or form['mode'] == '2':
        req_data['description'] += '\nfor_complete_events:%s\n' % (
            ','.join(["%s/%s" % (event['host'], event['service']) for event in events_for_downtime])
        )

    # ставим фильтры
    if form['mode'] == '0':
        req_data['filters'] = []
        for event in current_events.get():
            req_data['filters'].append(
                {
                    "host": event.host,
                    "service": event.name,
                }
            )
            if isinstance(event, CompleteEvent):
                req_data['filters'][-1]['namespace'] = event.namespace

    elif form['mode'] == '1':
        req_data['filters'] = [
            {
                "host": crit_raw_event['host'],
            } for event in current_events.get() for crit_raw_event in event.crit_raw_events
        ]

    elif form['mode'] == '2':
        req_data['filters'] = [
            {
                "host": crit_raw_event['host'],
                "service": crit_raw_event['service'],
            } for event in current_events.get() for crit_raw_event in event.crit_raw_events
        ]

    headers = {"Content-Type": "application/json",
               "Accept": "application/json",
               "Authorization": 'OAuth %s' % JUGGLER_TOKEN}

    data_from_api = json.loads(requests.post(MAKE_DOWNTIME_URL, data=json.dumps(req_data), headers=headers).content)

    response = {}
    if 'downtime_id' in data_from_api:
        try:
            add_comment_to_ticket(
                ticket_name,
                u"Ссылка на даунтайм в Juggler:\nhttps://juggler.yandex-team.ru/downtimes/?query=downtime_id=%s" % (
                    data_from_api['downtime_id']
                ),
                u"make_downtime"
            )
        except:
            pass
        response['success'] = True
        response['message'] = ticket_name
    else:
        response['success'] = False
        response['message'] = data_from_api['message']

    return jsonify(response)


@juggler_blueprint.route('/remove_downtime/')
@juggler_blueprint.route('/remove_downtime/<path:params>')
def remove_downtime(params=''):
    downtime_id = request.args.get('downtime_id', '')
    if downtime_id == '':
        return jsonify({'success': False, 'message': 'Error: empty "downtime_id" field'})

    downtimes_from_api = Downtimes(
        filters={'filters': [{'downtime_id': downtime_id}]},
        without_own_downtimes=True
    ).get()

    if not downtimes_from_api:
        return jsonify({'success': False, 'message': 'Error: downtime not found'})

    downtime = downtimes_from_api[0]
    req_data = {
        "downtime_ids": [downtime.id],
    }

    headers = {"Content-Type": "application/json",
               "Accept": "application/json",
               "Authorization": 'OAuth %s' % JUGGLER_TOKEN}

    try:
        data_from_api = json.loads(
            requests.post(REMOVE_DOWNTIME_URL, data=json.dumps(req_data), headers=headers).content
        )
    except:
        return jsonify({'success': False, 'message': 'Error: backend error during deleting downtime'})

    response = {}
    if 'downtimes' in data_from_api:
        response['success'] = True
    else:
        response['success'] = False
        response['message'] = data_from_api['message']

    if response['success'] and downtime.ticket:
        add_comment_to_ticket(
            downtime.ticket,
            u"Даунтайм %s был удален\nФильтры даунтайма:\n%s" % (downtime.id, downtime.filters_str),
            u"remove_downtime"
        )

    return jsonify(response)


@juggler_blueprint.route('/link_downtime/')
@juggler_blueprint.route('/link_downtime/<path:params>')
def link_downtime(params=''):
    downtime_id = request.args.get('downtime_id', '')
    ticket = request.args.get('ticket', '')
    days_to_rm = request.args.get('days_to_rm', MAX_DAYS_FOR_DOWNTIME)

    if downtime_id == '':
        return jsonify({'success': False, 'message': 'Error: empty "downtime_id" field'})

    downtimes_from_api = Downtimes(
        filters={'filters': [{'downtime_id': downtime_id}]}, without_own_downtimes=True
    ).get()

    if not downtimes_from_api:
        return jsonify({'success': False, 'message': 'Error: downtime not found'})

    downtime = downtimes_from_api[0]

    headers = {"Content-Type": "application/json",
               "Accept": "application/json",
               "Authorization": 'OAuth %s' % JUGGLER_TOKEN}

    try:
        req_data = {
            "downtime_id": downtime.id,
            "description": u'startrek:%s\ndays_to_rm:%s\n%s' % (
                ticket, days_to_rm, downtime.description
            ),
            "filters": downtime.filters,
            "start_time": time.time(),
            "end_time": downtime.end_time,
        }

        data_from_api = json.loads(
            requests.post(MAKE_DOWNTIME_URL, data=json.dumps(req_data), headers=headers).content
        )

    except:
        return jsonify({'success': False, 'message': 'Error: backend error during deleting downtime'})

    response = {}
    if 'downtime_id' in data_from_api:
        response['success'] = True
        response['message'] = ticket
        add_comment_to_ticket(
            ticket,
            u"К данному тикету привязан даунтайм %s:\nФильтры даунтайма:\n%s" % (
                downtime.id, downtime.filters_str
            ),
            u"link_downtime"
        )
    else:
        response['success'] = False
        response['message'] = data_from_api['message']

    return jsonify(response)


@juggler_blueprint.route('/get_tags/')
def get_tags():
    all_checks = juggler_api.get_checks([CheckFilter(namespace=DIRECT_NAMESPACE, host='', service='')])

    tags = {}
    for check in all_checks.checks:
        for tag in check.tags:
            if not tag.startswith('a_mark'):
                tags[tag] = 1

    return jsonify(sorted([tag for tag in tags]))


@juggler_blueprint.route('/check_def/')
@juggler_blueprint.route('/check_def/<path:params>')
def check_def(params=''):
    form = {
        'host': request.args.get('host_name', ''),
        'service': request.args.get('service_name', ''),
    }

    return jsonify(get_check_def(form['host'], form['service']))


@juggler_blueprint.route('/grafana/search', methods=['GET', 'POST', 'OPTIONS'])
def grafana_search():
    return grafana_response([])


@juggler_blueprint.route('/grafana/', methods=['GET', 'POST', 'OPTIONS'])
@juggler_blueprint.route('/grafanat/', methods=['GET', 'POST', 'OPTIONS'])
def grafana_ok():
    return grafana_response({"msg": "ok"})


@juggler_blueprint.route('/grafanat/search', methods=['GET', 'POST', 'OPTIONS'])
def grafana_search_tags():
    return grafana_response([])


@juggler_blueprint.route('/grafanat/query', methods=['GET', 'POST', 'OPTIONS'])
def grafana_query_tags():
    return grafana_query(True)


@juggler_blueprint.route('/grafana/query', methods=['GET', 'POST', 'OPTIONS'])
def grafana_query(use_tags=False):
    targets = []
    if request.data:
        for target in request.get_json(force=True)['targets']:
            targets.append(target['target'])

    if use_tags:
        events = CompleteEvents(
            hosts=[h['host'] for h in hosts['direct']],
            tags=targets,
            for_grafana=True
        ) if targets else CompleteEvents(from_list=[])
    else:
        events = CompleteEvents(hosts=targets, for_grafana=True) if targets else CompleteEvents(from_list=[])
    events.sort()

    series = []
    for event in events.get():
        series.append(event.as_grafana_array())

    data = [
        {
            "columns": [
                {
                    "text": "Failed",
                    "type": "string"
                },
                {
                    "text": "Name",
                    "type": "string"
                },
                {
                    "text": "S",
                    "type": "number"
                }
            ],
            "rows": series,
            "type": "table"
        }
    ]
    return grafana_response(data)


@juggler_blueprint.route('/grafanat/annotations', methods=['GET', 'POST', 'OPTIONS'])
@juggler_blueprint.route('/grafana/annotations', methods=['GET', 'POST', 'OPTIONS'])
def grafana_annotations():
    return grafana_response([])


@juggler_blueprint.route('/grafanat_hosts/query', methods=['GET', 'POST', 'OPTIONS'])
def grafana_hosts_query_tags():
    return grafana_hosts_query(True)


@juggler_blueprint.route('/grafana_hosts/query', methods=['GET', 'POST', 'OPTIONS'])
def grafana_hosts_query(use_tags=False):
    targets = []
    if request.data:
        for target in request.get_json(force=True)['targets']:
            targets.append(target['target'])

    if use_tags:
        events = CompleteEvents(
            hosts=[h['host'] for h in hosts['direct']],
            tags=targets,
            for_grafana=True
        ) if targets else CompleteEvents(from_list=[])
    else:
        events = CompleteEvents(hosts=targets, for_grafana=True) if targets else CompleteEvents(from_list=[])
    events.sort()

    series = []
    for event in events.get():
        series.append(event.as_grafana_array(with_host=True))

    data = [
        {
            "columns": [
                {
                    "text": "Failed",
                    "type": "string"
                },
                {
                    "text": "Host",
                    "type": "string"
                },
                {
                    "text": "Name",
                    "type": "string"
                },
                {
                    "text": "S",
                    "type": "number"
                }
            ],
            "rows": series,
            "type": "table"
        }
    ]
    return grafana_response(data)


@juggler_blueprint.route('/grafana/directadmintv/query', methods=['GET', 'POST', 'OPTIONS'])
def grafana_directadmintv():
    resp = requests.post(
        "http://juggler-api.search.yandex.net/v2/checks/get_checks_state",
        json={"filters": [{"tags": ["directadmin_TV"]}]}
    ).json()
    status2int = {"CRIT": 2, "WARN": 1, "OK": 0}
    series = [[item["host"], item["service"], status2int[item["status"]]] for item in resp["items"]]

    data = [
        {
            "columns": [
                {
                    "text": "Host",
                    "type": "string"
                },
                {
                    "text": "Name",
                    "type": "string"
                },
                {
                    "text": "S",
                    "type": "number"
                }
            ],
            "rows": series,
            "type": "table"
        }
    ]
    return grafana_response(data)
