# -*- coding: utf8 -*-

import os
import subprocess
import re
import tempfile
from startrek_client import Startrek
from flask import after_this_request, request, render_template, url_for, jsonify, Blueprint

from direct.infra.observatorium.lib.tools.settings import DIRECT_PPC_STARTREK_TOKEN

release_deps_blueprint = Blueprint('release_deps', __name__, url_prefix='/release_deps')

USER_AGENT = 'Direct.Observatorium'

deps_script = u"/usr/local/bin/dt-deps-manager"

# для дебага локально
DEBUG = False


@release_deps_blueprint.route('/')
@release_deps_blueprint.route('/index')
def index():
    return render_template('release_deps/release_deps_index.html')


@release_deps_blueprint.route('/ticket_viz')
@release_deps_blueprint.route('/ticket_viz/<path:params>')
def ticket_viz(params=''):
    client = Startrek(useragent=USER_AGENT,
                      token=DIRECT_PPC_STARTREK_TOKEN)

    ticket = request.args.get('ticket', '')
    releases = request.args.get('releases', '')

    if releases:
        return render_template('release_deps/release_deps_ticket_viz.html',
                               releases=releases,
                               title=u"Визуализация зависимостей в привязке к релизам")

    if not ticket:
        return render_template('release_deps/release_deps_ticket_viz.html', ticket=ticket,
                               error_message=u"Пустой параметр названия тикета")

    try:
        if ticket != 'all':
            client.issues[ticket]
    except:
        return render_template('release_deps/release_deps_ticket_viz.html', ticket=ticket,
                               error_message=u"Тикета '%s' не существует" % ticket)

    return render_template('release_deps/release_deps_ticket_viz.html',
                           ticket=ticket)


@release_deps_blueprint.route('/get_image_graph')
@release_deps_blueprint.route('/get_image_graph/<path:params>')
def get_image_graph(params=''):
    ticket = request.args.get('ticket', '')
    releases = request.args.get('releases', '')
    args = []
    upload_folder = tempfile.gettempdir()

    if releases:
        args.extend(["--release-visualize", "--one-image"])
    else:
        args.append("--visualize")

        if ticket == 'all':
            args.append("--one-image")
        else:
            args.extend(["--dependency", ticket])

    try:
        if DEBUG:
            output = subprocess.check_output(["ssh", "ppcdev2", deps_script] + args)
        else:
            args += ['--dir', upload_folder]
            output = subprocess.check_output([deps_script] + args)

        filename = re.search(ur"^File: (.+)$", output, re.I | re.M)

        if filename:
            filename = filename.group(1)
            if DEBUG:
                subprocess.call(["rsync", "ppcdev2:%s" % filename, upload_folder])
                filename = os.path.join(upload_folder, os.path.basename(filename))

            with open(filename, "rb") as image_file:
                image_data = image_file.read()

            result = {'image': image_data}
        else:
            result = {'error': 'Не удалось получить визуализацию зависимостей, вывод dt-deps-manager:\n%s' % output}

    except:
        result = {
            'error': 'Не удалось получить визуализацию зависимостей, подробности см. на ppcdev: dt-deps-manager -v'
        }

    @after_this_request
    def cleanup(response):
        if 'image' in result:
            os.remove(filename)
        return response

    return jsonify(result)


@release_deps_blueprint.route('/list_tickets/')
@release_deps_blueprint.route('/list_tickets/<path:params>')
def list_tickets(params=''):
    client = Startrek(useragent=USER_AGENT,
                      token=DIRECT_PPC_STARTREK_TOKEN)
    try:
        if DEBUG:
            output = subprocess.check_output(["ssh", "ppcdev2", deps_script, "-cod"])
        else:
            output = subprocess.check_output([deps_script, "-cod"])

    except subprocess.CalledProcessError as e:
        output = e.output

    tickets = re.findall(ur"(DIRECT-[0-9]+):\n(.+)$", output, re.I | re.M)

    url_viz = url_for('.ticket_viz')
    data = [{
        'id': ticket[0],
        'correct': ticket[1] == 'ok',
        'summary': client.issues[ticket[0]].summary,
        'link': "%s?ticket=%s" % (url_viz, ticket[0])
    } for ticket in tickets]

    return render_template('release_deps/release_deps_list_tickets.html',
                           tickets=data)
