# -*- coding: utf8 -*-

import os
from flask import render_template, send_file, after_this_request, request, Blueprint
import tempfile
from functools import cmp_to_key
from graphviz import Source
from kazoo.client import KazooClient
import requests

from direct.infra.observatorium.lib.tools.settings import SOLOMON_ROBOT_OAUTH_TOKEN

storages_blueprint = Blueprint('storages', __name__, url_prefix='/storages')

ZK_KWARGS = {
    'hosts': "ppcback01f.yandex.ru,ppcback01e.yandex.ru,ppcback01i.yandex.ru",
    'timeout': 10,
    'connection_retry': {"max_tries": 1, "delay": 1, "max_jitter": 1, "backoff": 1, "ignore_expire": False},
    'command_retry': {"max_tries": 1, "delay": 1, "max_jitter": 1, "backoff": 1, "ignore_expire": False},
}
ZOOKEEPER_LOCATION = '/direct/db-repl-status'
ZOOKEEPER_NODE = ZOOKEEPER_LOCATION + '/%s/graph'


@storages_blueprint.route('/index')
def index():
    return render_template('storages/storages_index.html')


@storages_blueprint.route('/get_replication_schema_graph')
def get_replication_schema_graph():
    zkh = KazooClient(**ZK_KWARGS)
    zkh.start()
    graph = zkh.get(ZOOKEEPER_NODE % request.args.get('instance', ''))[0]
    zkh.stop()

    graph_image = Source(graph, format='svg', engine='dot')
    with tempfile.NamedTemporaryFile(delete=False) as tmpfile:
        graph_image.render(tmpfile.name)

    @after_this_request
    def cleanup(response):
        os.remove(tmpfile.name)
        return response

    return send_file(tmpfile.name + '.svg', cache_timeout=5, mimetype='image/svg+xml')


@storages_blueprint.route('/replication_schema')
def replication_schema():
    zkh = KazooClient(**ZK_KWARGS)
    zkh.start()
    instances = sorted(
        zkh.get_children(ZOOKEEPER_LOCATION),
        key=cmp_to_key(
            lambda x, y: (
                (x > y) - (x < y)
                if x[:7] != 'ppcdata' or y[:7] != 'ppcdata' or len(x) == len(y)
                else len(x) - len(y)
            )
        )
    )
    zkh.stop()

    return render_template('storages/storages_replication_schema.html', instances=instances)


@storages_blueprint.route('/solomon_metadashboard')
def solomon_metadashboard():
    data = requests.get(
        "https://solomon.yandex-team.ru/api/v2/projects/direct/dashboards?pageSize=all",
        headers={'Authorization': 'OAuth ' + SOLOMON_ROBOT_OAUTH_TOKEN}
    ).json()

    dashboards = []
    for dashboard in data['result']:
        dash_params = requests.get(
            "https://solomon.yandex-team.ru/api/v2/projects/direct/dashboards/%s" % dashboard['id'],
            headers={'Authorization': 'OAuth ' + SOLOMON_ROBOT_OAUTH_TOKEN}
        ).json()['parameters']

        # https://st.yandex-team.ru/DIRECT-99846
        if dashboard['id'] == 'direct-yt-parametrized':
            for param in dash_params:
                if param['name'] == 'project':
                    dash_params.remove(param)
                    break
            for param in dash_params:
                if param['name'] == 'cluster':
                    param['value'] = 'hahn'
                elif param['name'] == 'account':
                    param['value'] = 'direct'

        url = "https://solomon.yandex-team.ru/?project=direct&dashboard=%s%s" % (
            dashboard['id'],
            "".join(["&%s=%s" % (param['name'], param['value']) for param in dash_params])
        )
        dashboards.append({
            'url': url,
            'name': dashboard['name']
        })

    return render_template('storages/solomon_metadashboard.html', dashboards=dashboards)
