# -* encoding: utf-8 -*-

import os
import datetime
import re
import json
from dateutil import tz

from direct.infra.observatorium.lib.tools.juggler import get_data_from_juggler_api, get_data_from_juggler_apiv2, nice_text, load_description_table

JUGGLER_HOSTS_FOR_COMPLETE_EVENTS = (
    'checks.direct.yandex.ru',
    'checks_day.direct.yandex.ru',
    'direct.apache',
)

JUGGLER_HOSTS_FOR_RAW_EVENTS = (
    'ppcback01e.yandex.ru',
    'ppcback01f.yandex.ru',
    'ppcback01i.yandex.ru',
)

# "важность" статуса, используется для сортировки событий
EVENTS_PRIORITY = {
    'CRIT': 9,
    'WARN': 5,
    'INFO': 3,
    'OK': 1,
}

EVENTS_SEVERITY = {
    'INFO': -1,
    'OK': 0,
    'WARN': 1,
    'CRIT': 2,
}

# Типы объектов, которые могут иметь downtime в juggler
DOWNTIME_OBJECT_TYPES = ('HOST', 'CGROUP', 'CMS')

WIKI_URL_FOR_DESCRIPTION = {
    'direct': 'direct/juggler-checks/checks-desc-table'
}

JUGGLER_USER = os.environ.get('JUGGLER_USER', 'ppc')


def ts_to_dt(ts):
    return datetime.datetime.utcfromtimestamp(ts).replace(tzinfo=tz.gettz('UTC')).astimezone(tz.gettz('Europe/Moscow'))


class RawEvent:
    def __init__(self, host, name, raw_event_data):
        data = raw_event_data.get('')
        self.host = host
        self.name = name
        self.hint = ''
        self.description = data['description'][0]['description']
        self.status = data['status'][0]
        self.priority = EVENTS_PRIORITY.get(self.status)
        self.time = int(data['status'][1])
        self.datetime = ts_to_dt(self.time).strftime("%Y-%m-%d %H:%M:%S")
        self.crit_raw_events = []

    def find_crit_raw_events(self, mode):
        self.crit_raw_events = [{'host': self.host, 'service': self.name}]

    def __unicode__(self):
        return self.name


class RawEvents:
    def __init__(self, with_obsolete=False, **kwargs):
        if with_obsolete:
            self.without_obsolete_flag = False
        else:
            self.without_obsolete_flag = True

        if 'from_list' in kwargs:
            self.events = kwargs['from_list']
        else:
            self.events = list()
            self.load_from_api(**kwargs)

    def load_from_api(self, **kwargs):
        raw_data = get_data_from_juggler_api(type='raw_events',
                                             without_obsolete=self.without_obsolete_flag,
                                             **kwargs
                                             )

        if raw_data == {}:
            return

        for host, data in raw_data.iteritems():
            if data:
                for name, raw_event_data in data.items():
                    self.events.append(RawEvent(host, name, raw_event_data))

    def sort(self):
        # Сортировка по-умолчанию - два прохода, сначала по имени
        self.events = sorted(self.events, key=lambda x: x.name)
        # потом обратная по приоритету и времени
        self.events = sorted(self.events, key=lambda x: (x.priority, x.time), reverse=True)

    def get(self):
        return self.events

    def filter_list_by_params(self, hosts={}, host_mode='', services={},
                              service_mode='', host_tags_filters=[], host_tags={}, form={}):
        new_events = []
        for new_event in self.events:
            if (not (host_mode == 'minus' and new_event.host not in hosts) and
                    not (host_mode == 'plus' and (not hosts or new_event.host in hosts))):
                continue

            if (not (service_mode == 'minus' and new_event.name not in services) and
                    not (service_mode == 'plus' and (not services or new_event.name in services))):
                continue

            match_filter = True
            for host_filter in host_tags_filters:
                if (not (host_filter[0] == '+' and host_tags[form['project']][new_event.host][host_filter[1:]]) and
                        not (host_filter[0] == '-' and not host_tags[form['project']][new_event.host][host_filter[1:]])):
                    match_filter = False
                    break

            if not match_filter:
                continue

            new_events.append(new_event)
        self.events = new_events

    def get_hints(self, project=''):
        if project not in WIKI_URL_FOR_DESCRIPTION:
            return
        description_table_raw_data = load_description_table(WIKI_URL_FOR_DESCRIPTION[project])
        description_table = {}
        for row in description_table_raw_data['data']['rows']:
            description_table[row[0]['raw']] = row[1]['raw']
        for event in self.events:
            event.hint = description_table[event.name] if event.name in description_table else ''


class CompleteEvent:
    def __init__(self, **kwargs):
        if kwargs.get("apiv2", False):
            self.host = kwargs["host"]
            self.name = kwargs["service"]
            self.description = kwargs["description"]
            self.status = kwargs["status"]
            self.timestamp = kwargs["change_time"]
            self.datetime = ts_to_dt(self.timestamp)
            self.date = self.datetime.strftime("%Y-%m-%d")
            self.time = self.datetime.strftime("%H:%M:%S")
            self.priority = EVENTS_PRIORITY.get(self.status)
            self.crit_raw_events = []
            self.hint = ''
            self.tickets = []
            self.failed_count = u""
            self.summary = u""
            self.status_begin_at = kwargs["change_time"]
            self.namespace = kwargs["namespace"]
            self.tags = {}

            return

        self.host = kwargs["host"]
        self.name = kwargs["name"]
        self.timestamp = int(kwargs["data"]['check_time'])
        self.datetime = ts_to_dt(self.timestamp)
        self.date = self.datetime.strftime("%Y-%m-%d")
        self.time = self.datetime.strftime("%H:%M:%S")
        self.namespace = ''
        self.status = kwargs["data"]['status'][0]
        self.status_begin_at = int(kwargs["data"]['status'][1])
        self.priority = EVENTS_PRIORITY.get(self.status)
        self.description = nice_text(kwargs["data"]['description'][0]['description'], field='description')
        summary = kwargs["data"]['description'][0].get('summary', '—')
        self.summary = nice_text(summary, field='summary')
        self.failed_count = summary.replace(' from ', '/').replace(' failed', '')
        self.tags = {tag: '' for tag in kwargs["data"]['tags']}
        self.crit_raw_events = []
        self.hint = ''
        self.tickets = []
        if (kwargs.get("need_children", False) and
                kwargs["data"]['description'][0] and
                'children' in kwargs["data"]['description'][0]):
            self.details = []
            for childrens_host, childrens_data in kwargs["data"]['description'][0]['children'].items():
                for children_name, one_children_data in childrens_data.items():
                    if '' in one_children_data:
                        status = one_children_data['']['status']
                        summary = one_children_data[''].get('summary', '-')
                    elif children_name in one_children_data:
                        status = one_children_data[children_name]['status']
                        summary = one_children_data[children_name].get('summary', '-')
                    else:
                        status = 'INFO'
                        summary = 'unknown instance'
                    self.details.append({
                        'host': childrens_host,
                        'name': children_name,
                        'status': status,
                        'summary': summary,
                    })
            self.details = sorted(self.details, key=lambda x: EVENTS_PRIORITY.get(x['status']), reverse=True)
        self.tmp = {}

    def __unicode__(self):
        return self.name

    def as_grafana_array(self, with_host=None):
        if with_host:
            return [
                self.failed_count,
                self.host,
                self.name,
                EVENTS_SEVERITY.get(self.status, 9),
            ]
        else:
            return [
                self.failed_count,
                self.name,
                EVENTS_SEVERITY.get(self.status, 9),
            ]

    def find_crit_raw_events(self, mode):
        self.tmp = {'mode': mode}
        self.find_crit_raw_events_dfs()
        self.crit_raw_events = sorted(self.crit_raw_events, key=lambda x: x['host'])

    def find_crit_raw_events_dfs(self, host=None, service=None):
        if not host or not service:
            host = self.host
            service = self.name

        if host in self.tmp and service in self.tmp[host]:
            return
        if host not in self.tmp:
            self.tmp[host] = {}
        self.tmp[host][service] = 1

        cur_event = CompleteEvents(hosts=[host], service_name=[service], need_children=True).get()

        cur_event = cur_event[0] if len(cur_event) == 1 else None

        if not cur_event or not hasattr(cur_event, 'details'):
            self.crit_raw_events.append({'host': host, 'service': service})
            return

        for child in cur_event.details:
            if child['status'] in [u'CRIT', u'WARN']:
                self.find_crit_raw_events_dfs(host=child['host'], service=child['name'])


class CompleteEvents:
    def __init__(self, **kwargs):
        if 'from_list' in kwargs:
            self.events = kwargs['from_list']
        else:
            self.events = list()
            if 'for_grafana' in kwargs:
                self.load_from_api_for_grafana(**kwargs)
            else:
                self.load_from_api(**kwargs)

    def load_from_api(self, **kwargs):
        if kwargs.get("apiv2", False):
            raw_data = get_data_from_juggler_apiv2(type='complete_events', **kwargs)

            for event in raw_data:
                self.events.append(CompleteEvent(apiv2=True, **event))

            return

        raw_data = get_data_from_juggler_api(type='complete_events',
                                             with_no_methods=True,
                                             old_juggler_format=False,
                                             **kwargs
                                             )

        if raw_data == {}:
            return

        for host, data in raw_data.iteritems():
            if data:
                for name, complete_event_data in data.items():
                    self.events.append(
                        CompleteEvent(
                            host=host,
                            name=name,
                            data=complete_event_data,
                            need_children=kwargs.get('need_children', False)
                        )
                    )

    def load_from_api_for_grafana(self, **kwargs):
        hosts_white = set()
        hosts_black = set()
        tags_white = set()
        tags_black = set()

        if 'hosts' in kwargs:
            for host in kwargs['hosts']:
                if host.startswith("!"):
                    hosts_black.add(host[1:])
                else:
                    hosts_white.add(host)

            kwargs['hosts'] = list(hosts_white)

        if 'tags' in kwargs:
            for tag in kwargs['tags']:
                if tag.startswith("!"):
                    tags_black.add(tag[1:])
                else:
                    tags_white.add(tag)

            kwargs['tags'] = list(tags_white)

        raw_data = get_data_from_juggler_api(type='complete_events',
                                             with_no_methods=True,
                                             old_juggler_format=False,
                                             **kwargs
                                             )

        if raw_data == {}:
            return

        for host, data in raw_data.iteritems():
            if hosts_white and host not in hosts_white:
                continue
            if hosts_black and host in hosts_black:
                continue
            if data:
                for name, complete_event_data in data.items():
                    if tags_white and len(tags_white.intersection(complete_event_data['tags'])) == 0:
                        continue
                    if tags_black and len(tags_black.intersection(complete_event_data['tags'])) > 0:
                        continue
                    self.events.append(CompleteEvent(host=host, name=name, data=complete_event_data))

    def sort(self):
        self.events = sorted(self.events, key=lambda x: (100 - x.priority, 0 if x.tickets else 1, x.host, x.name))

    def get(self):
        return self.events

    def filter_list_by_params(self, hosts={}, host_mode='', services={}, service_mode='',
                              host_tags_filters=[], host_tags={}, check_tags={}, form={}):
        new_events = []
        for new_event in self.events:
            if (not (host_mode == 'minus' and new_event.host not in hosts) and
                    not (host_mode == 'plus' and (not hosts or new_event.host in hosts))):
                continue

            if (not (service_mode == 'minus' and new_event.name not in services) and
                    not (service_mode == 'plus' and (not services or new_event.name in services))):
                continue

            match_filter = True
            for host_filter in host_tags_filters:
                if (not (host_filter[0] == '+' and host_tags[form['project']][new_event.host][host_filter[1:]]) and
                        not (host_filter[0] == '-' and not host_tags[form['project']][new_event.host][host_filter[1:]])):
                    match_filter = False
                    break

            if not match_filter:
                continue

            match_filter = True
            for check_filter in check_tags:
                if (not (check_filter[0] == '+' and check_filter[1:] in new_event.tags)
                        and (not (check_filter[0] == '-' and not check_filter[1:] in new_event.tags))):
                    match_filter = False
                    break

            if not match_filter:
                continue

            new_events.append(new_event)
        self.events = new_events

    def get_hints(self, project=''):
        if project not in WIKI_URL_FOR_DESCRIPTION:
            return
        description_table_raw_data = load_description_table(WIKI_URL_FOR_DESCRIPTION[project])
        description_table = {}
        for row in description_table_raw_data['data']['rows']:
            description_table[row[0]['raw']] = row[1]['raw']
        for event in self.events:
            event.hint = description_table[event.name] if event.name in description_table else ''

    def find_ticket_for_downtimes(self):
        my_downtimes = Downtimes(
            namespace=['direct.prod', 'direct.test', 'directmod.prod', 'bm.prod', 'partner.prod', 'advq.prod']
        ).get()

        dict_downtimes = {}

        for downtime in my_downtimes:
            if not downtime.ticket:
                continue

            for host in downtime.for_complete_events:
                for service in downtime.for_complete_events[host]:
                    if host not in dict_downtimes:
                        dict_downtimes[host] = {}

                    if service not in dict_downtimes[host]:
                        dict_downtimes[host][service] = []

                    dict_downtimes[host][service].append(downtime.ticket)

        for event in self.events:
            if event.host in dict_downtimes and event.name in dict_downtimes[event.host]:
                event.tickets = list(set(dict_downtimes[event.host][event.name]))
                continue

            if event.description == u'Forced OK by downtimes':
                downtime = Downtimes(hosts=[event.host], service_name=[event.name], without_own_downtimes=True)
                if len(downtime.get()) == 0 or downtime.get()[0].ticket == '':
                    continue
                event.tickets = [downtime.get()[0].ticket]


class Hosts:
    def __init__(self):
        self.hosts = dict()
        self.addDefaults()

    def addDefaults(self):
        for host in JUGGLER_HOSTS_FOR_RAW_EVENTS:
            self.hosts[host] = True

    def getAllHosts(self):
        return self.hosts

    def getCheckedHosts(self):
        result = []
        for host, checked in self.hosts.items():
            if checked:
                result.append(host)
        return result

    def applySessionData(self, data):
        if data and type(data).__name__ == 'dict':
            for host, checked in data.items():
                if checked:
                    self.hosts[host] = True
                else:
                    self.hosts[host] = False

    def applyUserData(self, data):
        for host in self.hosts.keys():
            self.hosts[host] = False
        if data:
            for host in set(data):
                if host:
                    self.hosts[host] = True
        if True not in self.hosts.values():
            self.addDefaults()


class Downtime:
    def __init__(self, **kwargs):
        self.filters = kwargs.get('filters')
        if self.filters:
            self.filters_str = json.dumps(self.filters, indent=2)
        self.start_time = int(kwargs.get('start_time'))
        self.start_datetime = ts_to_dt(self.start_time).strftime("%Y-%m-%d %H:%M:%S")
        self.description = kwargs.get('description')
        self.full_description = self.description
        self.user = kwargs.get('user')
        self.can_delete = self.user == JUGGLER_USER
        self.id = kwargs.get('downtime_id')
        self.source = kwargs.get('source')
        self.ticket = ''
        self.days_to_rm = ''
        self.for_complete_events = {}
        self.for_complete_events_str = ''

        if self.description:
            ticket = re.search('startrek:([A-Z-]+)([0-9]+)', self.description, re.I)
            if ticket and len(ticket.groups()) == 2:
                self.ticket = ticket.groups()[0] + ticket.groups()[1]
                self.description = self.description.replace(u'startrek:' + self.ticket, u'')

            result = re.search(r'for_complete_events:(.+)$', self.description, re.I)
            if result and len(result.groups()) == 1:
                self.for_complete_events_str = result.groups()[0]

                for complete_event in self.for_complete_events_str.split(','):
                    host = complete_event[:complete_event.index('/')]
                    service = complete_event[complete_event.index('/') + 1:]
                    if host not in self.for_complete_events:
                        self.for_complete_events[host] = {}

                    self.for_complete_events[host][service] = 1

                self.description = re.sub(r'for_complete_events:(.+)$', '', self.description)

            days_to_rm = re.search('days_to_rm:([0-9]+)', self.description, re.I)
            if days_to_rm and len(days_to_rm.groups()) == 1:
                self.days_to_rm = days_to_rm.groups()[0]
                self.description = self.description.replace(u'days_to_rm:' + self.days_to_rm, u'')

        if kwargs.get('end_time'):
            self.end_time = int(kwargs.get('end_time'))
            self.end_datetime = ts_to_dt(self.end_time).strftime("%Y-%m-%d %H:%M:%S")
        else:
            self.end_time = None
            self.end_datetime = u'∞'

    def __unicode__(self):
        return '<{0}:{1}> {2}'.format(self.object_type, self.object_name, self.service_name)


class Downtimes:
    def __init__(self, **kwargs):
        self.downtimes = list()

        self.object = ', '.join(kwargs.get('hosts', ''))
        self.service = ', '.join(kwargs.get('service_name', ''))
        self.filters = ', '.join(kwargs.get('filters', ''))

        self.load_from_api(**kwargs)

    def load_from_api(self, **kwargs):
        raw_data = get_data_from_juggler_apiv2(type='downtimes', **kwargs)
        if raw_data == {}:
            return

        for downtime in raw_data:
            self.downtimes.append(Downtime(**downtime))

    def sort(self):
        self.downtimes = sorted(self.downtimes, key=lambda x: x.start_time, reverse=True)

    def get(self):
        return self.downtimes
