# -* encoding: utf-8 -*-

import json
import string
import urllib
import urllib2
import itertools
import copy
import requests
import datetime
from flask import make_response

from startrek_client import Startrek

from direct.infra.observatorium.lib.tools.settings import DIRECT_PPC_STARTREK_TOKEN

DOWNTIME_TICKET_COMPONENT = 'juggler_downtime_guard'

JUGGLER_API_URL = 'http://juggler-api.search.yandex.net/api'
JUGGlER_APIV2_URL = 'http://juggler-api.search.yandex.net/v2'
USER_AGENT = 'Direct.Observatorium'
WIKI_API_URL = 'https://wiki-api.yandex-team.ru/_api/frontend/'
GET_CHECK_DEF_URL = 'http://juggler-api.search.yandex.net/api/checks/checks?do=1&host_name=%s&service_name=%s&include_children=1'
LIMIT_PER_PAGE = 1000
SIGN = u"----\nobservatorium/%s"

JUGGLER_CE_NICE_TEXT_RULES = {
    'description': [
        # graphite_threshold объединяет описания через |
        ('|', "\n"),
        # типовой путь в графите почти всегда содержит эти строки
        ('direct_one_min.db_configurations.production.', '_.'),
        ('one_min.direct.production.', '_.'),
    ],
    'summary': [
        (' from ', '/'),
    ],
}

PARAMS_TRANSLATION = {
    'hosts': 'host_name',
    'tags': 'tag_name',
    'status': 'status',
    'service_name': 'service_name',
    'object_name': 'object_name',
    'expand_groups': 'expand_groups',
    'namespace': 'namespace_name',
}

PARAMS_TRANSLATION_APIV2 = {
    'hosts': 'host',
    'tags': 'tags',
    'status': 'statuses',
    'service_name': 'service',
    'object_name': 'object_name',
    'expand_groups': 'expand_groups',
    'namespace': 'namespace',
}


def _process_filter_params(all_params, filters=[]):
    """
    из словаря all_params делает список туплов
    ( ('host_name', '...'), ('host_name', '...'), ('service_name', '...') )
    включает только параметры, указанные в filters
    """
    good_params = list()

    for filter_param in filters:
        if filter_param not in PARAMS_TRANSLATION:
            raise Exception('no param translation for {0}'.format(filter_param))

        if filter_param in all_params and type(all_params[filter_param]) is list and len(all_params[filter_param]):
            good_params.extend(map(lambda p: (PARAMS_TRANSLATION[filter_param], p), all_params[filter_param]))

    if not good_params:
        raise ValueError('There are no applicable conditions for filtering')

    return good_params


def _process_filter_params_apiv2(all_params, filters=[]):
    # есть указан filters, то берем его, иначе составляем вручную
    if 'filters' in all_params and 'filters' in filters:
        return all_params['filters']

    good_params = {}
    all_params_tmp = copy.deepcopy(all_params)

    if 'status' in all_params_tmp:
        if all_params_tmp['status']:
            good_params[PARAMS_TRANSLATION_APIV2['status']] = all_params_tmp['status']

        del all_params_tmp['status']

    all_params_tmp = dict(
        (PARAMS_TRANSLATION_APIV2[key], value)
        for key, value in all_params_tmp.iteritems() if value and key in filters
    )

    good_params['filters'] = [
        dict(itertools.izip(all_params_tmp.keys(), filter_name))
        for filter_name in itertools.product(*[all_params_tmp[filter_key] for filter_key in all_params_tmp])
    ]

    if not good_params:
        raise ValueError('There are no applicable conditions for filtering')

    return good_params


def get_data_from_juggler_api(*args, **kwargs):
    if 'type' not in kwargs:
        raise Exception('missing data type')
    else:
        pass

    params = [('do', '1')]  # выполнить запрос, а не получить документацию

    if kwargs['type'] == 'raw_events':
        method = 'events/raw_events'
        params.extend(_process_filter_params(kwargs, ['hosts', 'status', 'service_name']))

        if 'without_obsolete' in kwargs and kwargs['without_obsolete']:
            # включая события с истекшим TTL
            params.append(('without_obsolete_flag', '1'))

    elif kwargs['type'] == 'check_tags':
        method = 'checks/checks'
        params.extend(_process_filter_params(kwargs, ['hosts', 'service_name']))
        params.append(('include_children', '1'))

    elif kwargs['type'] == 'complete_events':
        method = 'events/complete_events'
        params.extend(_process_filter_params(kwargs, ['hosts', 'tags', 'status', 'service_name']))

        if 'with_no_methods' in kwargs and kwargs['with_no_methods']:
            # включая события без методов
            params.append(('all_flag', '1'))

    elif kwargs['type'] == 'downtimes':
        method = 'downtimes/downtimes'

        if 'service' in kwargs and kwargs['service_name']:
            params.append(('service_name', kwargs['service_name']))

        if 'object' in kwargs and kwargs['object']:
            params.append(('object_name', kwargs['object']))

        if 'expand_groups' in kwargs and kwargs['expand_groups']:
            params.append(('expand_flag', '1'))

    elif kwargs['type'] == 'downtimes_v2':
        method = 'downtimes_v2/get_downtime'

        if kwargs['service_name']:
            params.extend(_process_filter_params(kwargs, ['service_name']))

        if 'object' in kwargs and kwargs['object']:
            params.append(('object_name', kwargs['object']))

        if 'expand_groups' in kwargs and kwargs['expand_groups']:
            params.append(('expand_groups', '1'))

    elif kwargs['type'] == 'suggest_event_host':
        method = 'dashboard/suggest_checks'
        query = 'h@%s?' % kwargs.get('host', '')
        params.append(('query', query))

    elif kwargs['type'] == 'checks':
        method = 'checks/checks'
        params.extend(_process_filter_params(kwargs, ['namespace']))

    else:
        raise Exception('unknown data type')

    # Для использования старого Juggler'а группируем параметры по ключу
    if 'old_juggler_format' in kwargs and kwargs['old_juggler_format']:
        params_by_key = dict()
        for kv_pair in params:
            try:
                params_by_key[kv_pair[0]].add(kv_pair[1])
            except KeyError:
                params_by_key[kv_pair[0]] = set([kv_pair[1]])
        params = list((k, string.join(map(str, list(v)), ',')) for k, v in params_by_key.items())

    try:
        req = urllib2.Request('{0}/{1}?{2}'.format(JUGGLER_API_URL, method, urllib.urlencode(params)))
        req.add_header('User-agent', USER_AGENT)
        return json.loads(urllib2.urlopen(req).read())

    except:
        return {}


def get_data_from_juggler_apiv2(*args, **kwargs):
    if 'type' not in kwargs:
        raise Exception('missing data type')
    else:
        pass

    params = []

    if kwargs['type'] == 'downtimes':
        method = 'downtimes/get_downtimes'
        params = _process_filter_params_apiv2(kwargs, ['hosts', 'service_name', 'namespace', 'filters'])
        params['page_size'] = LIMIT_PER_PAGE
        params['page'] = 1
        if 'filters' in params and not kwargs.get("without_own_downtimes", False):
            params['filters'].append({'user': 'ppc'})

    elif kwargs['type'] == 'complete_events':
        method = 'checks/get_checks_state'
        params = _process_filter_params_apiv2(kwargs, ['tags', 'status', 'namespace', 'filters'])
    else:
        raise Exception('unknown data type')

    try:
        headers = {"Content-Type": "application/json",
                   "Accept": "application/json"}

        return json.loads(
            requests.post('%s/%s' % (JUGGlER_APIV2_URL, method), data=json.dumps(params), headers=headers).content
        )['items']

    except:
        return {}


def load_description_table(table_url=''):
    req = urllib2.Request(WIKI_API_URL + table_url + '/.grid?format=json')
    req.add_header('User-agent', USER_AGENT)
    req.add_header('Content-Type', 'application/json')
    req.add_header('Authorization', 'OAuth ' + DIRECT_PPC_STARTREK_TOKEN)
    return json.loads(urllib2.urlopen(req).read())


def nice_text(text, field=None):
    new_text = text
    if field in JUGGLER_CE_NICE_TEXT_RULES:
        for old, new in JUGGLER_CE_NICE_TEXT_RULES[field]:
            new_text = new_text.replace(old, new)
    return new_text


def grafana_response(data):
    response = make_response(json.dumps(data))
    response.headers['Access-Control-Allow-Headers'] = 'accept, content-type'
    response.headers['Access-Control-Allow-Methods'] = 'POST'
    response.mimetype = 'application/json'
    response.headers['Access-Control-Allow-Origin'] = '*'
    return response


def get_check_def(host, service):
    req = urllib2.Request(GET_CHECK_DEF_URL % (host, service))
    req.add_header('User-agent', USER_AGENT)
    req.add_header('Content-Type', 'application/json')

    try:
        response = json.loads(urllib2.urlopen(req).read())
    except:
        return {'Error': 'error in loading from api'}

    return response


def add_comment_to_ticket(ticket_name, comment, method):
    try:
        client = Startrek(useragent=u'direct-tag-cloud',
                          token=DIRECT_PPC_STARTREK_TOKEN)
        issue = client.issues[ticket_name]
        issue.comments.create(text=u"%s\n%s" % (comment, SIGN % method))

    except Exception as e:
        print e


def create_ticket(events, form, method, is_complete_event):
    client = Startrek(useragent=u'direct-tag-cloud',
                      token=DIRECT_PPC_STARTREK_TOKEN)

    description_string = u""

    for event in events:
        if is_complete_event:
            description_string += u'Агрегированное событие\nСтатус: %s %s\nПоследнее изменение статуса: %s\n%s' % (
                event.status,
                event.summary,
                datetime.datetime.fromtimestamp(event.status_begin_at).strftime("%Y-%m-%d %H:%M:%S"),
                u"https://juggler.yandex-team.ru/check_details/?host=%s&service=%s" % (event.host, event.name),
            )
        else:
            description_string += u'Сырое событие\nПоследнее изменение статуса: %s\n%s' % (
                datetime.datetime.fromtimestamp(event.time).strftime("%Y-%m-%d %H:%M:%S"),
                u"https://juggler.yandex-team.ru/raw_events/?query=%s" % (
                    urllib.quote_plus('host=' + event.host + '&service=' + event.name)
                )
            )

        if form['mode'] == '1':
            description_string += u'\n\n**Даунтайм поставлен только на хосты загоревшихся сырых событий!!**'
            description_string += u'\n\nПод даунтайм попали следующие хосты:\n%s' % (
                '\n'.join(["%s" % (crit_event['host']) for crit_event in event.crit_raw_events])
            )
        elif form['mode'] == '2':
            description_string += u'\n\n**Даунтайм поставлен только на хосты и сервисы загоревшихся сырых событий!!**'
            description_string += u'\n\nПод даунтайм попали следующие события:\n%s' % (
                '\n'.join([
                    "%s: %s" % (crit_event['host'], crit_event['service']) for crit_event in event.crit_raw_events
                ])
            )

        if hasattr(event, 'details') and form['mode'] == '0':
            description_string += u'\nДочерние:'
            for child in event.details:
                if child['status'] != u'OK':
                    description_string += u'\n' + child['status'] + u' ' + child['host'] + u': ' + child['name']

        description_string += '\n\n'

    issue = client.issues.create(
        queue=form['queue'],
        summary=(
            form['ticket_summary']
            if form['ticket_summary']
            else u'Починить juggler-проверку ' + ', '.join(["%s: %s" % (event.host, event.name) for event in events])
        ),
        type={'name': 'Task'},
        components=DOWNTIME_TICKET_COMPONENT,
        description=u"%s\n%s" % (description_string, SIGN % method),
    )

    return issue
