package main

import (
	"crypto/sha1"
	"crypto/tls"
	"crypto/x509"
	"encoding/hex"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"net/url"

	//"github.com/fatih/structs"
	ch "github.com/ClickHouse/clickhouse-go"
	"github.com/jmoiron/sqlx"
)

/* Программа создает структу таблиц, схожую с указанным кластером. */

func NewClientConnect(host, user, password, database string, port int, debug, compress, sslmode bool) Client {
	return Client{
		Host:     host,
		Username: user,
		Password: password,
		Port:     port,
		Debug:    debug,
		Database: database,
		Compress: compress,
		SSLMode:  sslmode,
	}
}

type Client struct {
	Host     string `structs:"-"`
	Port     int    `structs:"-"`
	Username string `structs:"username" json:"username"`
	Password string `structs:"password" json:"password"`
	Debug    bool   `structs:"debug" json:"debug"`
	Compress bool   `structs:"compress" json:"compress"`
	SSLMode  bool   `structs:"ssl" json:"ssl"`
	Database string `structs:"database" json:"database"`
}

func connect(client Client) (conn *sqlx.DB, err error) {
	urlValues := url.Values{}
	params := make(map[string]interface{})
	if data, err := json.Marshal(client); err != nil {
		return nil, fmt.Errorf("error marshal %v: %s", data, err)
	} else {
		err := json.Unmarshal(data, &params)
		if err != nil {
			return nil, fmt.Errorf("error unmarshal %s: %s", data, err)
		}
	}
	//params := structs.Map(client)
	for key, v := range params {
		var value string
		switch v1 := v.(type) {
		case int:
			value = fmt.Sprintf("%d", v1)
		case string:
			value = v1
		case bool:
			value = fmt.Sprintf("%t", v1)
		default:
			value = fmt.Sprintf("%v", v1)
		}
		if len(value) > 0 {
			urlValues.Add(key, value)
		}
	}

	if client.SSLMode {
		rootCertPool, err := x509.SystemCertPool()
		if err != nil {
			return nil, err
		}
		err = ch.RegisterTLSConfig("custom", &tls.Config{
			RootCAs: rootCertPool,
		})
		if err != nil {
			return nil, fmt.Errorf("error register tsl config: %s", err)
		}
	}

	var sock string
	if len(urlValues) > 0 {
		sock = fmt.Sprintf("tcp://%s:%d?%s", client.Host, client.Port, urlValues.Encode())
	} else {
		sock = fmt.Sprintf("tcp://%s:%d", client.Host, client.Port)
	}

	conn, err = sqlx.Open("clickhouse", sock)
	if err != nil {
		return
	}
	if err = conn.Ping(); err != nil {
		if exception, ok := err.(*ch.Exception); ok {
			err = fmt.Errorf("[%d] %s \n%s", exception.Code, exception.Message, exception.StackTrace)
		}
	}
	return
}

type TableMeta struct {
	Database    string `db:"database"`
	Name        string `db:"name"`
	Engine      string `db:"engine"`
	CreateQuery string `db:"create_table_query"`
}

func (tm TableMeta) Hash() string {
	sha := sha1.New()
	_, _ = sha.Write([]byte(tm.CreateQuery))
	return hex.EncodeToString(sha.Sum(nil))
}

func (tm TableMeta) Equal(table TableMeta) bool {
	return table.Hash() == tm.Hash()
}

func (tm TableMeta) FullName() string {
	return fmt.Sprintf("%s.%s", tm.Database, tm.Name)
}

type TablesMeta []TableMeta

func (tsm TablesMeta) FindTable(table TableMeta) (result TableMeta, ok bool) {
	for _, tm := range tsm {
		if table.Name == tm.Name && table.Database == tm.Database &&
			table.Engine == tm.Engine {
			return tm, true
		}
	}
	return
}

func (tsm TablesMeta) HasDatabase(db DatabaseMeta) (ok bool) {
	for _, tm := range tsm {
		if db.Name == tm.Database {
			return true
		}
	}
	return
}

func (tsm TablesMeta) DatabaseNames() (result []string) {
	tmp := make(map[string]int)
	for _, table := range tsm {
		tmp[table.Database] = 1
	}
	for dbname := range tmp {
		result = append(result, dbname)
	}
	return
}

func GetTablesMeta(conn *sqlx.DB) (result TablesMeta, err error) {
	query := "SELECT database, name, engine, create_table_query FROM system.tables WHERE database != 'system'"
	err = conn.Select(&result, query)
	return
}

type DatabaseMeta struct {
	Name   string `db:"name"`
	Engine string `db:"engine"`
}

type DatabasesMeta []DatabaseMeta

func GetDatabaseMeta(conn *sqlx.DB) (result DatabasesMeta, err error) {
	query := "SELECT name, engine FROM system.databases WHERE name != 'system'"
	err = conn.Select(&result, query)
	return
}

func CreateTable(conn *sqlx.DB, table TableMeta) (err error) {
	_, err = conn.Exec(table.CreateQuery)
	return
}

func CreateDatabase(conn *sqlx.DB, db DatabaseMeta) (err error) {
	query := fmt.Sprintf("CREATE DATABASE IF NOT EXISTS %s ENGINE=%s", db.Name, db.Engine)
	_, err = conn.Exec(query)
	return
}

func main() {
	srcdb := flag.String("src", "localhost", "source database")
	dstdb := flag.String("dst", "localhost", "destination database")
	user := flag.String("user", "default", "connect user")
	password := flag.String("password", "", "connect password")
	port := flag.Int("port", 9000, "connect port")
	database := flag.String("database", "default", "database name")
	debug := flag.Bool("debug", false, "debug mode")
	compress := flag.Bool("compress", true, "enable/disable compress")
	sslmode := flag.Bool("ssl", false, "enable ssl mode")
	doit := flag.Bool("doit", false, "run to create tables")

	flag.Parse()

	var missedTables, wrongStuctTables TablesMeta

	srcClient := NewClientConnect(*srcdb, *user, *password, *database, *port, *debug, *compress, *sslmode)
	destClient := NewClientConnect(*dstdb, *user, *password, *database, *port, *debug, *compress, *sslmode)

	srcConn, err := connect(srcClient)
	if err != nil {
		log.Fatal(err)
	}
	dstConn, err := connect(destClient)
	if err != nil {
		log.Fatal(err)
	}

	srcDatabases, err := GetDatabaseMeta(srcConn)
	if err != nil {
		log.Fatal(err)
	}
	if len(srcDatabases) == 0 {
		log.Fatalf("not found databases in %s\n", srcClient.Host)

	}

	srcTables, err := GetTablesMeta(srcConn)
	if err != nil {
		log.Fatal(err)
	}

	dstTables, err := GetTablesMeta(dstConn)
	if err != nil {
		log.Fatal(err)
	}

	for _, sTable := range srcTables {
		if dTable, ok := dstTables.FindTable(sTable); ok {
			if dTable.Hash() != sTable.Hash() {
				wrongStuctTables = append(wrongStuctTables, sTable)
			}
		} else {
			missedTables = append(missedTables, sTable)
		}
	}

	if len(missedTables) > 0 {
		log.Printf("missed tables: \n")
		for _, table := range missedTables {
			log.Printf("\t%s (%s)\n", table.FullName(), table.Engine)
			if *debug {
				log.Printf("\t\t%s\n", table.CreateQuery)
			}
		}
	}

	if len(wrongStuctTables) > 0 {
		fmt.Printf("wroong struct tables: \n")
		for _, table := range missedTables {
			log.Printf("\t%s (%s)\n", table.FullName(), table.Engine)
			if *debug {
				log.Printf("\t\t%s\n", table.CreateQuery)
			}
		}
	}

	if (*doit) && len(missedTables) > 0 {
		log.Printf("start create missed tables...\n")
		for _, db := range srcDatabases {
			if !dstTables.HasDatabase(db) {
				if err := CreateDatabase(dstConn, db); err != nil {
					log.Printf("[critical] error create database %s: %s\n", db.Name, err)
				}
			}
		}
		for _, table := range missedTables {
			if err := CreateTable(dstConn, table); err != nil {
				log.Printf("[critical] error create table %s: %s\n", table.FullName(), err)
			}
		}
		log.Printf("finish create missed tables\n")
	}

	if len(missedTables) == 0 && len(wrongStuctTables) == 0 {
		log.Printf("all tables good\n")
	}
}
