package main

import (
	"bytes"
	"fmt"
	"time"
)

//Сравниваем UUID от response и request запросов. На каждый ответ должен приходить хотябы один запрос.

const (
	BSREQUEST string = "SELECT extract(metadata,'uuid=([\\\\w-]*),?') AS uuid, extract(source,'file:([\\\\w-.]*)/?') AS hostname FROM bsexport_data_lr_distributed" +
		" WHERE data_type='request' AND (log_date=toDate(%[1]d) OR log_date=toDate(%[2]d)) AND" +
		" toDateTime(%[1]d)<=log_time AND log_time<toDateTime(%[2]d) GROUP BY uuid, hostname FORMAT JSON"
	BSRESPONSE string = "SELECT extract(metadata,'uuid=([\\\\w-]*),?') AS uuid, extract(source,'file:([\\\\w-.]*)/?') AS hostname FROM bsexport_data_lr_distributed" +
		" WHERE data_type='response' AND (log_date=toDate(%[1]d) OR log_date=toDate(%[2]d)) AND" +
		" toDateTime(%[1]d)<=log_time AND log_time<toDateTime(%[2]d) GROUP BY uuid, hostname FORMAT JSON"
)

type ChResponse struct {
	Data UUIDHosts `json:"data"`
	Rows int       `json:"rows"`
}

type UUIDHost struct {
	UUID     string `json:"uuid"`
	Hostname string `json:"hostname"`
}

type UUIDHosts []UUIDHost

func (u UUIDHosts) find(fhost UUIDHost) bool {
	for _, uhost := range u {
		if (uhost.UUID == fhost.UUID) && (uhost.Hostname == fhost.Hostname) {
			return true
		}
	}
	return false
}

type BsStat struct {
	lost     UUIDHosts
	found    UUIDHosts
	err      error
	code     int
	requests []string
}

func (stat BsStat) LogFormat(verbose bool) string {
	result := bytes.NewBufferString("")
	const GREEN = "\033[92mOK\033[0m"
	const RED = "\033[91mFAIL\033[0m"

	if stat.err != nil {
		return fmt.Sprintf("%s %s", RED, stat.err)
	}

	if stat.code != 0 {
		for _, uhost := range stat.lost {
			msg := fmt.Sprintf("%s lost %s on %s\n", RED, uhost.UUID, uhost.Hostname)
			result.WriteString(msg)
		}
	}

	if verbose {
		for _, request := range stat.requests {
			msg := fmt.Sprintln(request)
			result.WriteString(msg)
		}
		for _, uhost := range stat.found {
			msg := fmt.Sprintf("%s found %s on %s\n", GREEN, uhost.UUID, uhost.Hostname)
			result.WriteString(msg)
		}
	}
	return result.String()
}

func (stat BsStat) MonrunFormat() (int, string) {
	if stat.code != 0 {
		msg := fmt.Sprintf("[CheckBsexportStat] lost %d bsexport UUID requests;", len(stat.lost))
		return 1, msg
	}
	msg := "[CheckBsexportStat] OK;"
	return 0, msg
}

func (ts *Tasks) CheckBsexportStat(ch ClickHouse, params interface{}) Status {
	var bsResponse, bsRequest ChResponse
	var code = 0
	var foundUUID, lostUUID UUIDHosts
	var requests []string

	cnf := params.(map[interface{}]interface{})
	intervalTime := cnf["intervalTime"].(string)
	delta, _ := time.ParseDuration(intervalTime)

	nowTime := time.Now()
	endCheckTime := nowTime.Add(-12 * time.Hour)
	startCheckTimeResponse := endCheckTime.Add(-delta)
	startCheckTimeRequest := startCheckTimeResponse.Add(-60 * time.Minute)

	qRequest := fmt.Sprintf(BSREQUEST, startCheckTimeRequest.Unix(), endCheckTime.Unix())
	qResponse := fmt.Sprintf(BSRESPONSE, startCheckTimeResponse.Unix(), endCheckTime.Unix())
	requests = append(requests, qRequest, qResponse)
	//logger.Print(qRequest)
	//logger.Print(qResponse)

	if err := ch.executeJSON(qRequest, &bsRequest); err != nil {
		return BsStat{err: err, code: 1}
	}

	if err := ch.executeJSON(qResponse, &bsResponse); err != nil {
		return BsStat{err: err, code: 1}
	}

	dataBsResponse := bsResponse.Data
	dataBsRequest := bsRequest.Data

	for _, uhost := range dataBsResponse {
		//fmt.Println(uhost)
		if dataBsRequest.find(uhost) {
			foundUUID = append(foundUUID, uhost)
		} else {
			lostUUID = append(lostUUID, uhost)
		}
	}

	if len(lostUUID) != 0 {
		code = 1
	}

	return BsStat{found: foundUUID, lost: lostUUID, err: nil, code: code, requests: requests}
}
