package main

import (
	"bytes"
	"crypto/tls"
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"os"
	"reflect"
	"sync"
	"time"

	"gopkg.in/yaml.v2"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

const (
	Host   string = "ppchouse-cloud.direct.yandex.net"
	Port   int    = 8443
	User   string = "direct_reader"
	Token  string = "/etc/direct-tokens/clickhouse_direct_reader"
	DB     string = "directdb"
	Config string = "/etc/dt-monitoring-loadlogs/main.yaml"
)

type Status interface {
	MonrunFormat() (int, string)
	LogFormat(bool) string
}

type ClickHouse struct {
	address  string
	user     string
	database string
	port     int
	name     string
	password string
}

func NewClickHouse(address, user, database string, port int, name, token string) (ClickHouse, error) {
	data, err := ioutil.ReadFile(token)
	if err != nil {
		return ClickHouse{}, err
	}
	data = bytes.Trim(data, "\n")
	return ClickHouse{
		address,
		user,
		database,
		port,
		name,
		string(data),
	}, nil
}

func (ch ClickHouse) Socket() string {
	return fmt.Sprintf("%s:%d", ch.address, ch.port)
}

func (ch ClickHouse) execute(request string) (data []byte, err error) {
	var response *http.Response
	logger.Debug("execute request: %s", request)
	conn, err := net.DialTimeout("tcp", ch.Socket(), time.Duration(2*time.Second))
	if err != nil {
		msg := fmt.Sprintf("error connect to %s", ch.address)
		return nil, errors.New(msg)
	} else {
		Wrap(conn.Close())
	}

	tr := &http.Transport{
		TLSClientConfig:     &tls.Config{InsecureSkipVerify: true},
		MaxIdleConnsPerHost: 10,
		MaxIdleConns:        10,
	}
	client := http.Client{
		Timeout:   time.Duration(120 * time.Second),
		Transport: tr,
	}

	body := url.Values{"query": {request},
		"database": {ch.database},
	}

	myapi := fmt.Sprintf("https://%s/?%s", ch.Socket(), body.Encode())
	logger.Debug("connect: %s", myapi)

	req, _ := http.NewRequest("GET", "https://"+ch.Socket()+"/?"+body.Encode(), nil)
	req.Header.Set("X-ClickHouse-User", ch.user)
	req.Header.Set("X-ClickHouse-Key", ch.password)
	response, err = client.Do(req)
	if err == nil {
		defer func() { _ = response.Body.Close() }()
		data, err = ioutil.ReadAll(response.Body)
		if response.StatusCode == 200 {
			return data, err
		} else {
			return nil, fmt.Errorf("wrong response. Status code %s %s", response.Status, data)
		}
	}
	return nil, fmt.Errorf("error request %s: %s", request, err)
}

func (ch ClickHouse) executeJSON(request string, schema interface{}) (err error) {
	var out []byte
	if out, err = ch.execute(request); err != nil {
		return err
	}
	if err = json.Unmarshal(out, schema); err != nil {
		return err
	}
	logger.Debug("executeJSON: %+v", schema)
	return nil
}

type Tasks struct{}

func (ch ClickHouse) run(nameCheck string, tasks Tasks, wg *sync.WaitGroup, results chan Status, params interface{}) {
	defer wg.Done()
	args := []reflect.Value{reflect.ValueOf(ch), reflect.ValueOf(params)}
	out := reflect.ValueOf(&tasks).MethodByName(nameCheck).Call(args)
	results <- out[0].Interface().(Status)
}

func MonrunStatus(data chan Status) string {
	result := bytes.NewBufferString("")
	var code int
	var helper string
	for status := range data {
		c, msg := status.MonrunFormat()
		result.WriteString(msg)
		if c > code {
			code = c
		}
	}
	if code > 0 {
		helper = fmt.Sprintf("Run command: '%s [-v]'", os.Args[0])
	}
	msg := fmt.Sprintf("%d; %s %s", code, result.String(), helper)
	return msg
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}

func main() {
	var wg sync.WaitGroup
	var config map[string]interface{}
	var raw []byte
	var err error

	var debugMode, verboseMode bool
	var monrunMode bool
	var port int
	var confPath, tokenFile, database, user, host string
	flag.BoolVar(&verboseMode, "v", false, "verbose mode")
	flag.BoolVar(&monrunMode, "m", false, "monrun mode")
	flag.StringVar(&confPath, "config", Config, "config path")
	flag.BoolVar(&debugMode, "debug", false, "debug mode")
	flag.StringVar(&tokenFile, "token", Token, "password for connect")
	flag.StringVar(&database, "database", DB, "database for connect")
	flag.StringVar(&user, "user", User, "user for connect")
	flag.StringVar(&host, "host", Host, "host for connect")
	flag.IntVar(&port, "port", Port, "port for connect")
	flag.Parse()

	logger.NewLoggerFormat(
		logger.WithLogLevel(logger.InfoLvl),
		logger.WithLogger(log.New(os.Stdout, "", log.LstdFlags)))

	if debugMode {
		logger.WithLogLevel(logger.DebugLvl)
	}

	if raw, err = ioutil.ReadFile(confPath); err != nil {
		logger.Crit("%s", err)
		os.Exit(1)
	}
	if err = yaml.Unmarshal(raw, &config); err != nil {
		logger.Crit("%s", err)
		os.Exit(2)
	}

	dataStatus := make(chan Status, 200)

	var tasks Tasks
	for check, params := range config {
		savedCheck, savedParams := check, params
		connect, err := NewClickHouse(host, user, database, port, check, tokenFile)
		if err != nil {
			logger.Crit("error NewClickHouse: %s", err)
			os.Exit(0)
		}
		go connect.run(savedCheck, tasks, &wg, dataStatus, savedParams)
		wg.Add(1)
	}

	wg.Wait()
	close(dataStatus)

	if monrunMode {
		fmt.Println(MonrunStatus(dataStatus))
		return
	}
	for status := range dataStatus {
		fmt.Printf("%s", status.LogFormat(verboseMode))
	}
}
