package main

import (
	"bytes"
	"fmt"
	"strconv"
	"strings"
	"sync"
)

type MaxLineStat struct {
	name      string
	column    string
	size      int64
	threshold int64
	code      int
	err       error
}

type MaxLineStats []MaxLineStat

func (mls MaxLineStats) MonrunFormat() (int, string) {
	warn := bytes.NewBufferString("")
	for _, metric := range mls {
		var msg string
		if metric.code == 0 {
			continue
		}
		if metric.err == nil {
			msg = fmt.Sprintf("%s/%s(%d);", metric.name, metric.column, metric.size)
		} else {
			msg = fmt.Sprintf("%s/%s(%s);", metric.name, metric.column, strings.Trim(metric.err.Error(), "\n"))
		}
		warn.WriteString(msg)
	}
	if warn.Len() != 0 {
		result := fmt.Sprintf("[MaxLineStats] %s", warn.String())
		return 1, result
	}
	result := "[MaxLineStats] OK;"
	return 0, result
}

func (mls MaxLineStats) LogFormat(verbose bool) string {
	result := bytes.NewBufferString("")
	const GREEN = "\033[92mOK\033[0m"
	const RED = "\033[91mFAIL\033[0m"
	for _, metric := range mls {
		var msg string
		color := GREEN
		if (metric.code == 0) && (!verbose) {
			continue
		}
		if metric.code > 0 {
			color = RED
		}
		if metric.err == nil {
			msg = fmt.Sprintf("%s [MaxLineStats] %s column '%s': %d bytes (duration %d)\n", color, metric.name, metric.column, metric.size, metric.threshold)
		} else {
			msg = fmt.Sprintf("%s [MaxLineStats] %s column '%s': %s\n", color, metric.name, metric.column, metric.err.Error())
		}
		result.WriteString(msg)
	}
	return result.String()
}

func (ts *Tasks) MaxLineStat(ch ClickHouse, params interface{}) Status {
	var sections []map[string]string
	var result MaxLineStats
	for _, s := range params.([]interface{}) {
		section := make(map[string]string)
		for key, val := range s.(map[interface{}]interface{}) {
			k := fmt.Sprintf("%s", key)
			v := fmt.Sprintf("%v", val)
			section[k] = v
		}
		sections = append(sections, section)
	}

	var wg sync.WaitGroup
	templQuery := "SELECT max(length(%[1]s)) AS size FROM %[2]s WHERE log_time > toDateTime('%[3]s') AND log_time < toDateTime('%[4]s') AND log_date >= toDate('%[3]s') AND log_date <= toDate('%[4]s') FORMAT JSON"
	chanMaxLineStat := make(chan MaxLineStat, 200)
	threadsNum := make(chan int, 8)

	for _, s := range sections {
		threadsNum <- 1
		var tmp ResponseStaticStat
		var code int

		threshold, _ := strconv.ParseInt(s["threshold"], 10, 64)
		name := s["name"]
		column := s["column"]

		minTime, maxTime := intervalTime(s["duration"])
		query := fmt.Sprintf(templQuery, column, name, minTime, maxTime)

		wg.Add(1)
		go func(string, int64, *sync.WaitGroup, chan MaxLineStat, chan int) {
			defer wg.Done()
			defer func() {
				<-threadsNum
			}()
			if err := ch.executeJSON(query, &tmp); err != nil {
				chanMaxLineStat <- MaxLineStat{name: name, column: column, code: 1, err: err}
				return
			}
			size, _ := strconv.ParseInt(tmp.Data[0]["size"], 10, 64)

			if size > threshold {
				code = 1
			}

			chanMaxLineStat <- MaxLineStat{name, column, size, threshold, code, nil}
		}(query, threshold, &wg, chanMaxLineStat, threadsNum)
	}
	wg.Wait()
	close(chanMaxLineStat)
	close(threadsNum)

	for metric := range chanMaxLineStat {
		result = append(result, metric)
	}
	return result
}
