package main

import (
	"bytes"
	"fmt"
	"strconv"
	"strings"
	"sync"
	"time"
)

type Table struct {
	Name      string `yaml:"name"`
	Duration  string `yaml:"duration"`
	Threshold int64  `yaml:"threshold"`
}

type ResponseStaticStat struct {
	Data []map[string]string `json:"data"`
	Rows int                 `json:"rows"`
}

func intervalTime(interval string) (string, string) {
	now := time.Now()
	checktime := now.Add(-60 * time.Minute)
	duration, _ := time.ParseDuration(interval)
	before := checktime.Add(-duration)
	return before.Format("2006-01-02 15:04:05"), now.Format("2006-01-02 15:04:05")
}

type StaticStat struct {
	name      string
	threshold int64
	count     int64
	code      int
	err       error
}

type StaticStats []StaticStat

func (ss StaticStats) LogFormat(verbose bool) string {
	result := bytes.NewBufferString("")
	const GREEN = "\033[92mOK\033[0m"
	const RED = "\033[91mFAIL\033[0m"
	for _, metric := range ss {
		var msg string
		color := GREEN
		if (metric.code == 0) && (!verbose) {
			continue
		}
		if metric.code > 0 {
			color = RED
		}
		if metric.err == nil {
			msg = fmt.Sprintf("%s [CheckStaticStat] %s: %d (need more %d)\n", color, metric.name, metric.count, metric.threshold)
		} else {
			msg = fmt.Sprintf("%s [CheckStaticStat] %s: %s\n", color, metric.name, metric.err.Error())
		}
		result.WriteString(msg)
	}
	return result.String()
}

func (ss StaticStats) MonrunFormat() (int, string) {
	warn := bytes.NewBufferString("")
	for _, metric := range ss {
		var msg string
		if metric.code == 0 {
			continue
		}
		if metric.err == nil {
			msg = fmt.Sprintf("%s(%d);", metric.name, metric.count)
		} else {
			msg = fmt.Sprintf("%s(%s);", metric.name, strings.Trim(metric.err.Error(), "\n"))
		}
		warn.WriteString(msg)
	}
	if warn.Len() != 0 {
		result := fmt.Sprintf("[CheckStaticStat] %s", warn.String())
		return 1, result
	}
	result := "[CheckStaticStat] OK;"
	return 0, result
}

func (ts *Tasks) CheckStaticStat(ch ClickHouse, params interface{}) Status {
	var sections []map[string]string
	var result StaticStats
	for _, s := range params.([]interface{}) {
		section := make(map[string]string)
		for key, val := range s.(map[interface{}]interface{}) {
			k := fmt.Sprintf("%s", key)
			v := fmt.Sprintf("%v", val)
			section[k] = v
		}
		sections = append(sections, section)
	}
	var wg sync.WaitGroup
	templQuery := "SELECT count() AS count FROM %[1]s WHERE log_time > toDateTime('%[2]s') AND log_time < toDateTime('%[3]s') AND log_date >= toDate('%[2]s') AND log_date <= toDate('%[3]s') FORMAT JSON"
	chanStaticStat := make(chan StaticStat, 200)
	threadsNum := make(chan int, 8)
	for _, s := range sections {
		threadsNum <- 1
		var tmp ResponseStaticStat
		var code int

		threshold, _ := strconv.ParseInt(s["threshold"], 10, 64)
		name := s["name"]

		minTime, maxTime := intervalTime(s["duration"])
		query := fmt.Sprintf(templQuery, name, minTime, maxTime)

		wg.Add(1)
		go func(string, int64, *sync.WaitGroup, chan StaticStat, chan int) {
			defer wg.Done()
			defer func() {
				<-threadsNum
			}()
			if err := ch.executeJSON(query, &tmp); err != nil {
				chanStaticStat <- StaticStat{name: name, code: 1, err: err}
				return
			}
			count, _ := strconv.ParseInt(tmp.Data[0]["count"], 10, 64)

			if count < threshold {
				code = 1
			}

			chanStaticStat <- StaticStat{name, threshold, count, code, nil}
		}(query, threshold, &wg, chanStaticStat, threadsNum)
	}
	wg.Wait()
	close(chanStaticStat)
	close(threadsNum)

	for metric := range chanStaticStat {
		result = append(result, metric)
	}
	return result
}
