package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
	"syscall"
	"time"

	yaml "gopkg.in/yaml.v2"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

const (
	DefaultConfig  = "/etc/yandex/statbox-push-client/push-client.yaml"
	DefaultCommand = "/usr/bin/push-client -c /etc/yandex/statbox-push-client/push-client.yaml --status --json"
)

var (
	configPushClient        string
	monrunFormat, debugMode bool
)

type Config struct {
	Files FilesStatus
}

type FileStatus struct {
	Name     string    `yaml:"name" db:"name"`
	Ident    string    `yaml:"ident" db:"ident"`
	LogType  string    `yaml:"log_type" db:"log_type"`
	Inode    uint64    `yaml:"-" db:"inode"`
	Size     int64     `yaml:"-" db:"size"`
	TimeRead time.Time `yaml:"-" db:"log_time"`
}

type FilesStatus []FileStatus

func (fs FileStatus) Stat() (sysstat *syscall.Stat_t, err error) {
	var fd *os.File
	var fileinfo os.FileInfo
	if fd, err = os.OpenFile(fs.Name, os.O_RDONLY, 0444); err != nil && os.IsNotExist(err) {
		return nil, err
	} else if err != nil {
		return nil, fmt.Errorf("error read file %s: %s", fs.Name, err)
	}
	defer func() { _ = fd.Close() }()
	if fileinfo, err = fd.Stat(); err != nil {
		return nil, fmt.Errorf("error get stat file %s: %s", fs.Name, err)
	}
	sysstat, ok := fileinfo.Sys().(*syscall.Stat_t)
	if !ok {
		return nil, fmt.Errorf("error read stat file %s: %+v", fs.Name, sysstat)
	}
	return sysstat, nil
}

func ParseFileStatus(data []byte, dataStruct interface{}) (fs FilesStatus, err error) {
	var errmsg []error
	var newfs FilesStatus

	switch v := dataStruct.(type) {
	case Config:
		if err = yaml.Unmarshal(data, &v); err != nil {
			return fs, fmt.Errorf("[LoadPushClient] unmarshal %s: %s", data, err)
		}
		newfs = v.Files
	case FilesStatus:
		if err = yaml.Unmarshal(data, &v); err != nil {
			return fs, fmt.Errorf("[LoadPushClient] unmarshal %s: %s", data, err)
		}
		newfs = v
	default:
		return nil, fmt.Errorf("dont found supported types")
	}

	if len(newfs) == 0 {
		return fs, fmt.Errorf("[LoadPushClient] empty files list from %v", fs)
	}

	for _, file := range newfs {
		if stat, err := file.Stat(); err == nil {
			newFile := FileStatus{
				Name:  file.Name,
				Inode: stat.Ino,
				Size:  stat.Size,
			}
			fs = append(fs, newFile)
		} else if os.IsNotExist(err) {
			logger.Debug("not found file %s, skipit", file.Name)
			continue
		} else {
			errmsg = append(errmsg, err)
		}
	}
	if len(errmsg) > 0 {
		return fs, fmt.Errorf("[LoadPushClient] %v", errmsg)
	}
	return fs, nil
}

func LoadConfigPushClient(path string) (files FilesStatus, err error) {
	var mystr Config
	var data []byte
	if data, err = ioutil.ReadFile(path); err != nil {
		return files, fmt.Errorf("[LoadPushClient] error read %s: %s", path, err)
	}
	return ParseFileStatus(data, mystr)
}

func LoadStatusPushClient() (files FilesStatus, err error) {
	var pushStatus FilesStatus
	cmd := strings.Split(DefaultCommand, " ")
	data, err := exec.Command(cmd[0], cmd[1:]...).Output()
	if err != nil {
		return files, fmt.Errorf("[LoadStatusPushClient] error get stat %s: %s", DefaultCommand, err)
	}
	if err = json.Unmarshal(data, &pushStatus); err != nil {
		return files, fmt.Errorf("[LoadStatusPushClient] unmarshal %s: %s", data, err)
	}
	if len(pushStatus) == 0 {
		return files, fmt.Errorf("[LoadStatusPushClient] empty files list from %s", DefaultCommand)
	}
	return ParseFileStatus(data, pushStatus)
}

func (fs FilesStatus) Find(value FileStatus) (FileStatus, error) {
	for _, f := range fs {
		if f.Name == value.Name {
			return f, nil
		}
	}
	return FileStatus{}, fmt.Errorf("not found Name '%s'", value.Name)
}

func main() {
	flag.StringVar(&configPushClient, "config", DefaultConfig, "config push-client")
	flag.BoolVar(&monrunFormat, "monrun", false, "monrun format")
	flag.BoolVar(&debugMode, "debug", false, "debug mode")
	flag.Parse()

	logger.NewLoggerFormat(
		logger.WithLogLevel(logger.InfoLvl),
		logger.WithLogger(log.New(os.Stdout, "", log.LstdFlags)))
	if debugMode {
		logger.WithLogLevel(logger.DebugLvl)
	}

	files, err := LoadConfigPushClient(configPushClient)
	if err != nil {
		msg := fmt.Sprintf("error read config %s: %s", configPushClient, err)
		if monrunFormat {
			fmt.Printf("2; found errors. Run: '%s' for more information\n", os.Args[0])
		} else {
			logger.Crit(msg)
		}
		os.Exit(0)
	}

	pushStat, err := LoadStatusPushClient()
	if len(pushStat) == 0 {
		msg := "empty files json in push-client --status"
		if monrunFormat {
			fmt.Printf("0; %s", msg)
		} else {
			logger.Crit(msg)
		}
	}
	if err != nil {
		msg := fmt.Sprintf("error load status %s: %s", configPushClient, err)
		if monrunFormat {
			fmt.Printf("2; found errors. Run: '%s' for more information\n", os.Args[0])
		} else {
			logger.Crit(msg)
		}
		os.Exit(0)
	}

	var errmsg []error
	for _, f1le := range files {
		stat, err := pushStat.Find(f1le)
		if err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		if stat.Name == f1le.Name {
			logger.Debug("%s(inode: %d) %s(inode: %d)", stat.Name, stat.Inode, f1le.Name, f1le.Inode)
			if stat.Inode != f1le.Inode {
				errmsg = append(errmsg, err)
				continue
			}
		}
	}

	if len(errmsg) != 0 {
		msg := fmt.Sprintf("error load status %s: %s", configPushClient, err)
		if monrunFormat {
			fmt.Printf("2; found errors. Run: '%s' for more information\n", os.Args[0])
		} else {
			logger.Crit(msg)
		}
	} else {
		if monrunFormat {
			fmt.Printf("0; not found errors")
		} else {
			logger.Info("not found errors")
		}
	}
}
