package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"path/filepath"
	"regexp"
	"syscall"

	"gopkg.in/yaml.v2"
)

type FileStat struct {
	Inode    uint64 `json:"inode"`
	Name     string `json:"name"`
	Size     int64  `json:"size"`
	AggrName string `json:"aggrname"`
}

func (f FileStat) getInode() uint64 {
	return f.Inode
}

type FilePosition struct {
	FileStat
	LastPosition int64       `json:"position"`
	LogChannel   chan string `json:"-"`
}

func (fp *FilePosition) getFileName() string {
	fileStat := fp.FileStat
	return fileStat.Name
}

func (fp *FilePosition) getAggrName() string {
	fileStat := fp.FileStat
	return fileStat.AggrName
}

type FileStats []FileStat

func (fp FilePositions) getLastPosition(f FileStat) int64 {
	for _, p := range fp {
		if f.Inode == p.Inode && f.Name == p.Name && f.AggrName == p.AggrName {
			return p.LastPosition
		}
	}
	return 0
}

func (f FileStats) createChannels() map[string]chan []byte {
	var mp = make(map[string]chan []byte)
	for _, filestat := range f {
		if _, ok := mp[filestat.AggrName]; !ok {
			mp[filestat.AggrName] = make(chan []byte)
		}
	}
	return mp
}

func (f FileStats) NewFilePositions(saved FilePositions, logchan chan string) FilePositions {
	var filePositions FilePositions
	for _, filestat := range f {
		pos := saved.getLastPosition(filestat)
		filePosition := FilePosition{FileStat: filestat, LastPosition: pos, LogChannel: logchan}
		filePositions = append(filePositions, filePosition)
	}
	return filePositions
}

type FilePositions []FilePosition

func (fp FilePositions) getInodes() []uint64 {
	var inodes []uint64
	for _, p := range fp {
		inodes = append(inodes, p.getInode())
	}
	return inodes
}

type Config struct {
	StatusPath  string `yaml:"status_path"`
	LockPath    string `yaml:"lock_path"`
	dbPositions FilePositions
	Logs        map[string]string `yaml:"logs"`
}

func NewConfig(path string) (config Config) {
	var readBytes []byte
	var err error
	if readBytes, err = ioutil.ReadFile(path); err != nil {
		log.Fatal(err)
		return Config{}
	}
	if err = yaml.Unmarshal(readBytes, &config); err != nil {
		log.Fatal(err)
	}
	return
}

func (c *Config) lockPath() string {
	return c.LockPath
}

func (c *Config) loadDB(logchan chan string) {
	var readBytes []byte
	var err error
	if readBytes, err = ioutil.ReadFile(c.StatusPath); err != nil {
		logchan <- fmt.Sprintln(err)
		return
	}
	if err = json.Unmarshal(readBytes, &c.dbPositions); err != nil {
		log.Print(err)
	}
}

func (c *Config) savedDB(data FilePositions, logchan chan string) {
	var writeBytes []byte
	var err error
	if writeBytes, err = json.Marshal(data); err != nil {
		logchan <- fmt.Sprintln(err)
		return
	}
	if err = ioutil.WriteFile(c.StatusPath, writeBytes, 0644); err != nil {
		logchan <- fmt.Sprintln(err)
	}
}

func NewFileStats(c Config) (filestats FileStats) {
	for aggrLog, rangeLogs := range c.Logs {
		path, rgxp := filepath.Dir(rangeLogs), filepath.Base(rangeLogs)
		listing := getFilesStatus(path, rgxp, aggrLog)
		filestats = append(filestats, listing...)
	}
	return
}

func getFilesStatus(path, filter, aggr string) (result FileStats) {
	fileInfoLogs, err := ioutil.ReadDir(path)
	if err != nil {
		log.Print(err)
		return
	}
	rgxp := regexp.MustCompile(filter)
	for _, fileInfo := range fileInfoLogs {
		var ok bool
		var sysstat *syscall.Stat_t
		if fileInfo.IsDir() {
			continue
		}
		if ok = rgxp.MatchString(fileInfo.Name()); !ok {
			continue
		}
		if sysstat, ok = fileInfo.Sys().(*syscall.Stat_t); !ok {
			log.Print(err)
			continue
		}
		fullname := filepath.Join(path, fileInfo.Name())
		filestat := FileStat{Inode: sysstat.Ino, Name: fullname, Size: fileInfo.Size(), AggrName: aggr}
		result = append(result, filestat)
	}
	return
}
