package main

import (
	"bufio"
	"bytes"
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"
	//	"github.com/pkg/profile"
)

const (
	CONFIG = "/etc/yandex-dt-multitail/config.yaml"
	LOGGER = "/var/log/dt-multitail/main.log"
)

type Task struct {
	chanData   chan []byte
	lockWriter *Mutex
	lockReader *Mutex
	wgReader   *sync.WaitGroup
	wgWriter   *sync.WaitGroup
	sgWriter   chan os.Signal
	sgReader   chan os.Signal
	metaInfo   *FilePosition
}

func (t *Task) getTaskName() string {
	filePosition := t.metaInfo
	return filePosition.getFileName()
}

func (t *Task) getTaskAggrName() string {
	filePosition := t.metaInfo
	return filePosition.getAggrName()
}

func (t *Task) getTaskInode() uint64 {
	filePosition := t.metaInfo
	return filePosition.getInode()
}

func (t *Task) getLogChan() chan string {
	filePosition := t.metaInfo
	return filePosition.LogChannel
}

func (t *Task) getTaskLastPosition() int64 {
	filePosition := t.metaInfo
	return filePosition.LastPosition
}

func (t *Task) setTaskMetaSize(value int64) {
	filePosition := t.metaInfo
	filePosition.Size = value
}

func (t *Task) sumTaskMetaLastPosition(value int64) {
	filePosition := t.metaInfo
	filePosition.LastPosition += value
}

func (t *Task) writerLock() bool {
	logchan := t.getLogChan()
	if ok := t.lockWriter.TryLock(); !ok {
		logchan <- fmt.Sprintf("INFO [writertLock] lock blocked %s %t", t.getTaskAggrName(), ok)
		return false
	}
	return true
}

func (t *Task) writerUnlock() {
	t.lockWriter.Unlock()
}

func (t *Task) readerLock() bool {
	logchan := t.getLogChan()
	if ok := t.lockReader.TryLock(); !ok {
		logchan <- fmt.Sprintf("INFO [readerLock] lock blocked %s %t", t.getTaskName(), ok)
		return false
	}
	return true
}

func (t *Task) readerUnlock() {
	t.lockReader.Unlock()
}

func (t *Task) writer() {
	var fd *os.File
	var err error
	var shortErr bool

	//блокировка для исключения запущенных процессов
	if ok := t.writerLock(); !ok {
		return
	}
	defer t.writerUnlock()

	var aggrName = t.getTaskAggrName()
	//общий счетчик по работающим процессам на запись
	t.wgWriter.Add(1)
	defer t.wgWriter.Done()

	//ловим HUP сигнал для переоткрытия файлов на запись
	signal.Notify(t.sgWriter, syscall.SIGHUP)

	logchan := t.getLogChan()
	resend := make(chan []byte, 20000)
	defer close(resend)

K:
	for {
		var closing bool

		logchan <- fmt.Sprintf("INFO [writer] open %s, data channel %v, signal writer %v, %v", t.getTaskAggrName(), t.chanData, t.sgWriter, &t.wgWriter)

		if fd, err = os.OpenFile(t.getTaskAggrName(), os.O_WRONLY|os.O_CREATE|os.O_APPEND, 0644); err != nil {
			logchan <- fmt.Sprintf("INFO [writer] error open file %s: %s", aggrName, err)
			return
		}

		//идем в конец файла
		if _, err := fd.Seek(0, 2); err != nil {
			logchan <- fmt.Sprintf("[WARN] error seek file %s: %s", t.getTaskAggrName(), err)
		}

		//пишем на диск через buffer
		wrbuf := bufio.NewWriterSize(
			fd,
			4096*10,
		)

		if shortErr {
			shortErr = false
			if fileinfo, err := fd.Stat(); err == nil {
				if fileinfo.Size() > 0 {
					if _, err = wrbuf.Write([]byte{'\n'}); err != nil {
						logchan <- fmt.Sprintf("[WARN] error write to file %s: %s", t.getTaskAggrName(), err)
					}
				}
			} else {
				logchan <- fmt.Sprintf("WARN [writer] (%s) error stat file: %s", t.getTaskAggrName(), err)
			}
		}

		//закрываем файл, когда не остается readers
		finish := make(chan int)
		go func() {
			t.wgReader.Wait()
			close(finish)
		}()

		ticker := time.NewTicker(60 * time.Second)
		go func(*bufio.Writer) {
			for range ticker.C {
				if err := wrbuf.Flush(); err != nil {
					logchan <- fmt.Sprintf("WARN [writer] error flush file %s: %s", t.getTaskAggrName(), err)
				}
			}
		}(wrbuf)

		for {
			select {
			case line, ok := <-t.chanData:
				if !ok {
					logchan <- fmt.Sprintf("WARN [writer] (%s) exit writer because channel closed", aggrName)
					ticker.Stop()
					Wrap(wrbuf.Flush())
					Wrap(fd.Close())
					break K
				}
				if !bytes.HasSuffix(line, []byte{'\n'}) {
					logchan <- fmt.Sprintf("WARN [writer] (%s) broken line: %s", aggrName, line)
					break
				}
				if _, err = wrbuf.Write(line); err != nil {
					logchan <- fmt.Sprintf("ERROR [writer] (%s) error write line: %s ", aggrName, err)
					if err == io.ErrShortWrite {
						logchan <- fmt.Sprintf("ERROR [writer] (%s) error short line: %s", aggrName, line)
						resend <- line
						shortErr = true
						Wrap(wrbuf.Flush())
						Wrap(fd.Close())
						continue K
					}
				}
			case line := <-resend:
				logchan <- fmt.Sprintf("ERROR [writer] (%s) resend line: %s", aggrName, line)
				if _, err = wrbuf.Write(line); err != nil {
					logchan <- fmt.Sprintf("ERROR [writer] (%s) resend error: %s", aggrName, err)
					logchan <- fmt.Sprintln(line)
				}
			case _, ok := <-finish:
				if !ok && !closing {
					logchan <- fmt.Sprintf("WARN [writer] (%s) close channel from because all readers died", aggrName)
					closing = true
					close(t.chanData)
				}
			case <-t.sgWriter:
				logchan <- fmt.Sprintf("INFO [writer] (%s) recv hup signal", aggrName)
				ticker.Stop()
				if err := wrbuf.Flush(); err != nil {
					logchan <- fmt.Sprintf("WARN [writer] error flush file %s: %s", t.getTaskAggrName(), err)
				}
				if err := fd.Close(); err != nil {
					logchan <- fmt.Sprintf("WARN [writer] error close file %s: %s", t.getTaskAggrName(), err)
				}
				logchan <- fmt.Sprintf("INFO [writer] close %s", aggrName)
				continue K
			default:
				time.Sleep(5 * time.Second)
			}
		}
	}
	logchan <- fmt.Sprintf("WARN [writer] close %s", aggrName)
}

func (t *Task) reader() {
	var fd *os.File
	var err error

	//блокировка для исключения запущенных процессов
	if ok := t.readerLock(); !ok {
		return
	}
	defer t.readerUnlock()

	t.wgReader.Add(1)
	defer t.wgReader.Done()

	logchan := t.getLogChan()

	for {
		if ok := t.writerLock(); !ok {
			break
		}
		t.writerUnlock()
		logchan <- fmt.Sprintf("INFO [reader] (%s) waiting start writer thread", t.getTaskName())
		time.Sleep(5 * time.Second)
	}

	//ловим все TERM сигналы
	signal.Notify(t.sgReader, syscall.SIGTERM, syscall.SIGINT)

	logchan <- fmt.Sprintf("INFO [reader] open %s, data channel %v, signal writer %v", t.getTaskName(), t.chanData, t.sgReader)

	if fd, err = os.OpenFile(t.getTaskName(), os.O_RDONLY, 0444); err != nil {
		logchan <- fmt.Sprintf("CRIT [reader] (%s) error open file: %s", t.getTaskName(), err)
		return
	}
	defer func() { _ = fd.Close() }()

	if _, err := fd.Seek(t.getTaskLastPosition(), 0); err != nil {
		logchan <- fmt.Sprintf("CRIT [reader] (%s) error seek file: %s", t.getTaskName(), err)
		return
	}

	rbuf := bufio.NewReader(fd)

L:
	for {
		var line []byte

		if fileinfo, err := fd.Stat(); err != nil {
			logchan <- fmt.Sprintf("WARN [reader] (%s) error stat file: %s", t.getTaskName(), err)
		} else {
			t.setTaskMetaSize(fileinfo.Size())
		}

		for {
			var data []byte
			var err error

			select {
			case sg := <-t.sgReader:
				logchan <- fmt.Sprintf("WARN [reader] (%s) recv TERM signal %v", t.getTaskName(), sg)
				break L
			default:
			}

			if data, err = rbuf.ReadBytes('\n'); (err != nil) && (err != io.EOF) {
				logchan <- fmt.Sprintf("CRIT [reader] (%s) err read rbufs: %s", t.getTaskName(), err)
				break
			}
			if len(data) == 0 {
				time.Sleep(10 * time.Second)
				continue
			}
			line = append(line, data...)
			if !bytes.HasSuffix(line, []byte{'\n'}) {
				continue
			}
			t.chanData <- line
			sendLen := int64(len(line))
			line = []byte{}
			t.sumTaskMetaLastPosition(sendLen)
		}
	}
	logchan <- fmt.Sprintf("CRIT [reader] (%s) close", t.getTaskName())
}

func NewTask(fp *FilePosition, ts Tasks) Task {
	aggrName := fp.getAggrName()
	logchan := fp.LogChannel
	task := Task{metaInfo: fp,
		chanData:   ts.getChanData(aggrName),
		lockWriter: ts.getLockWriter(aggrName),
		lockReader: &Mutex{},
		wgReader:   ts.getWgReader(aggrName),
		wgWriter:   ts.getWgWriter(),
		sgWriter:   ts.getSgWriter(aggrName),
		sgReader:   make(chan os.Signal, 1),
	}
	logchan <- fmt.Sprintf("INFO [newtask] %v", task)
	return task
}

type Tasks []Task

func (ts Tasks) remove(indexes []int) Tasks {
	var cnt, num int
	for cnt, num = range indexes {
		ts[num] = ts[len(ts)-cnt-1]
	}
	return ts[:len(ts)-cnt-1]
}

func (ts Tasks) getWgWriter() *sync.WaitGroup {
	for _, task := range ts {
		if task.wgWriter != nil {
			return task.wgWriter
		}
	}
	var wg sync.WaitGroup
	return &wg
}

func (ts Tasks) getWgReader(name string) *sync.WaitGroup {
	for _, task := range ts {
		if (task.getTaskAggrName() == name) && (task.wgReader != nil) {
			return task.wgReader
		}
	}
	var wgr sync.WaitGroup
	return &wgr
}

func (ts Tasks) getChanData(name string) chan []byte {
	for _, task := range ts {
		if (task.getTaskAggrName() == name) && (task.chanData != nil) {
			return task.chanData
		}
	}
	ch := make(chan []byte, 100000)
	return ch
}

func (ts Tasks) getLockWriter(name string) *Mutex {
	for _, task := range ts {
		if (task.getTaskAggrName() == name) && (task.lockWriter != nil) {
			return task.lockWriter
		}
	}
	mutex := &Mutex{}
	return mutex
}

func (ts Tasks) getSgWriter(name string) chan os.Signal {
	for _, task := range ts {
		if (task.getTaskAggrName() == name) && (task.sgWriter != nil) {
			return task.sgWriter
		}
	}
	s1gnal := make(chan os.Signal, 1)
	return s1gnal
}

func (ts Tasks) equals(fp FilePosition) bool {
	for _, task := range ts {
		meta := task.metaInfo
		filestat := fp.FileStat
		if meta.getInode() == filestat.getInode() {
			return true
		}
	}
	return false
}

func (ts *Tasks) addReaderTasks(newfp *FilePositions) {
	for _, fp := range *newfp {
		filepos := fp
		logchan := fp.LogChannel
		if ok := ts.equals(filepos); !ok {
			task := NewTask(&filepos, *ts)
			logchan <- fmt.Sprintf("INFO [addReaderTasks] create new task %s", task.getTaskName())
			go task.writer()
			go task.reader()
			*ts = append(*ts, task)
		}
	}
}

func (ts *Tasks) removeReaderTasks(newfp *FilePositions) {
	inodes := newfp.getInodes()
	var indexes []int
	for num, task := range *ts {
		var founded = false
		var logchan = task.getLogChan()
		for _, inode := range inodes {
			if inode != task.getTaskInode() {
				continue
			}
			founded = true
		}
		if !founded {
			logchan <- fmt.Sprintf("INFO [removeOldTasks] remove %s", task.getTaskName())
			select {
			case task.sgReader <- syscall.SIGTERM:
			default:
			}
			indexes = append(indexes, num)
		}
	}
	if len(indexes) != 0 {
		*ts = ts.remove(indexes)
	}
}

func (ts *Tasks) reloadWriterThreads() {
	for _, t := range *ts {
		task := t
		logchan := task.getLogChan()
		logchan <- fmt.Sprintf("INFO [reloadWriterThreads] reload task %v", task)

		if ok := t.readerLock(); ok {
			t.readerUnlock()
			continue
		}
		time.Sleep(5 * time.Second)
		go t.writer()
	}
}

func (ts *Tasks) getAllMeta() (fp FilePositions) {
	for _, task := range *ts {
		meta := task.metaInfo
		fp = append(fp, *meta)
	}
	return
}

func startLogger(logchan chan string, logwg *sync.WaitGroup) {
	var fd *os.File
	var err error

	logwg.Add(1)
	defer logwg.Done()

	var reload = make(chan os.Signal, 1)
	signal.Notify(reload, syscall.SIGHUP)

P:
	for {
		if fd, err = os.OpenFile(LOGGER, os.O_WRONLY|os.O_CREATE|os.O_SYNC|os.O_APPEND, 0644); err != nil {
			fmt.Printf("error %s: %s", LOGGER, err)
		}
		logger := log.New(fd, "", log.LstdFlags)
		for {
			select {
			case logline, ok := <-logchan:
				if !ok {
					msg := fmt.Sprintln("logger thread stop because log channel closed.")
					logger.Print(msg)
					Wrap(fd.Close())
					break P
				}
				logger.Print(logline)
			case <-reload:
				msg := fmt.Sprintf("INFO [logger] (%s) recv hup signal", LOGGER)
				logger.Print(msg)
				Wrap(fd.Close())
				continue P
			default:
				time.Sleep(10 * time.Second)
			}
		}
	}
}

func main() {
	var configPath string
	var lockFd *os.File
	var err error

	flag.StringVar(&configPath, "f", CONFIG, "config path")
	flag.Parse()

	var stopflag = false

	var logchan = make(chan string, 200000)
	var logwg sync.WaitGroup
	go startLogger(logchan, &logwg)

	conf := NewConfig(configPath)
	conf.loadDB(logchan)

	if lockFd, err = os.OpenFile(conf.lockPath(), os.O_WRONLY|os.O_CREATE, 0755); err != nil {
		log.Fatalf("ERROR [main] error open lock file %s: %s\n", conf.lockPath(), err)
	}
	if err := syscall.Flock(int(lockFd.Fd()), syscall.LOCK_EX|syscall.LOCK_NB); err != nil {
		log.Fatalf("ERROR [main] dont lock file %s: %s\n", conf.lockPath(), err)
	}
	defer func() { _ = lockFd.Close() }()
	defer func() { _ = syscall.Flock(int(lockFd.Fd()), syscall.LOCK_UN) }()

	logchan <- fmt.Sprintf("INFO [main] config %v", conf)

	fileStat := NewFileStats(conf)
	filePositions := fileStat.NewFilePositions(conf.dbPositions, logchan)

	logchan <- fmt.Sprintf("INFO [main] start with config %v", filePositions)

	var tasks Tasks

	for _, fp := range filePositions {
		v := fp
		task := NewTask(&v, tasks)
		tasks = append(tasks, task)
	}

	//получаем
	wg := tasks.getWgWriter()

	for _, task := range tasks {
		t := task
		go t.writer()
		go t.reader()
	}

	var lockSavedDB *Mutex = &Mutex{}
	var lockMonitor *Mutex = &Mutex{}
	var lockGenerateTasks *Mutex = &Mutex{}

	//ловим SIGTERN/SIGINT для корректного завершения программы
	var sgmain = make(chan os.Signal, 1)
	signal.Notify(sgmain, syscall.SIGTERM, syscall.SIGINT)

	for {
		go func(*Tasks, *Config, chan string, *bool, *Mutex) {
			if ok := lockSavedDB.TryLock(); !ok {
				return
			}
			defer lockSavedDB.Unlock()

			ticker := time.NewTicker(5 * time.Second)
			defer ticker.Stop()

			for range ticker.C {
				conf.savedDB(tasks.getAllMeta(), logchan)
				if stopflag {
					return
				}
			}
		}(&tasks, &conf, logchan, &stopflag, lockSavedDB)

		go func(*Tasks, chan string, *bool, *Mutex) {
			if ok := lockMonitor.TryLock(); !ok {
				return
			}
			defer lockMonitor.Unlock()

			ticker := time.NewTicker(60 * time.Second)
			defer ticker.Stop()

			for range ticker.C {
				if stopflag {
					return
				}
				logchan <- fmt.Sprintf("INFO [active tasks] %v", tasks.getAllMeta())
			}
		}(&tasks, logchan, &stopflag, lockMonitor)

		go func(*Tasks, chan string, *bool, *Mutex) {
			if ok := lockGenerateTasks.TryLock(); !ok {
				return
			}
			defer lockGenerateTasks.Unlock()

			for {
				logchan <- "INFO [generate tasks] start round"
				if stopflag {
					return
				}
				newFileStat := NewFileStats(conf)
				newFilePositions := newFileStat.NewFilePositions(nil, logchan)
				tasks.removeReaderTasks(&newFilePositions)
				tasks.addReaderTasks(&newFilePositions)
				time.Sleep(5 * time.Second)
			}
		}(&tasks, logchan, &stopflag, lockGenerateTasks)

		select {
		case sg := <-sgmain:
			logchan <- fmt.Sprintf("INFO [main] ending program. Waiting writed threads. %s", sg)
			wg.Wait()
			logchan <- "INFO [main] end program"
			stopflag = true
			time.Sleep(7 * time.Second)
			close(logchan)
			logwg.Wait()
		default:
			time.Sleep(5 * time.Second)
		}
	}
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}
