# coding: utf8
"""
Проверяет наличие графиков в Golovan для Директовых gencfg групп
"""
import argparse
import json
import logging
import requests
import sys
import os
import time
import yaml

from infra.yasm.yasmapi import GolovanRequest


def get_gencfg_prj_geo(group):
    geo = group[:3].lower()

    url = 'http://api.gencfg.yandex-team.ru/trunk/groups/%s/card' % group
    res = requests.get(url).json()
    prj = res['tags']['prj']

    return prj, geo


def get_yasm_signals(prj, geo, periods_ago=1, period=3600):
    logger.info('fetching golovan data from -%d period (%ds) for prj %s, geo %s' % (periods_ago, period, prj, geo))

    et = int(time.time() / period) * period - period * periods_ago
    st = et - period

    host = 'ASEARCH'
    signals = [
        'itype=ppcvm;ctype=prod;prj={};geo={}:portoinst-cpu_usage_cores_tmmv'.format(prj, geo),
        'itype=ppcvm;ctype=prod;prj={};geo={}:portoinst-memory_usage_gb_tmmv'.format(prj, geo),
        'itype=ppcvm;ctype=prod;prj={};geo={}:portoinst-cpu_limit_cores_tmmv'.format(prj, geo),
        'itype=ppcvm;ctype=prod;prj={};geo={}:portoinst-memory_limit_gb_tmmv'.format(prj, geo),
    ]
    res = ()
    try:
        res = GolovanRequest(host, period, st, et, signals)
    except Exception as e:
        logger.error('cannot fetch data from golovan: %s %s' % (type(e), e))

    # ((1548928800, {'itype=ppcvm;ctype=prod;prj=direct-java-api5;geo=man:portoinst-cpu_limit_cores_tmmv': 394.7688798611115, 'itype=ppcvm...': ..., ...}), ...)
    return res


def yasm_data_exists(prj, geo):
    if isinstance(prj, list) or isinstance(prj, tuple):
        prj = ','.join(prj)
    logger.info('get yasm data from '
        'https://yasm.yandex-team.ru/chart/itype=ppcvm;hosts=ASEARCH;ctype=prod;prj=%s;geo=%s;signals=portoinst-cpu_usage_cores_tmmv' % (prj, geo))

    nodata_results = []
    for periods_ago in (1, 24, 24*7):
        for ts, data in get_yasm_signals(prj, geo, periods_ago):
            # считаем, что если все сенсоры == 0 за данный период, то данных там нет
            no_data = all([x == 0.0 for x in data.values()])
            nodata_results.append(no_data)
    total_nodata = all(nodata_results)
    return not total_nodata


def gencfg_group_yasm_data_exists(group):
    logger.info('get prj and geo from gencfg group ' + group)
    try:
        prj, geo = get_gencfg_prj_geo(group)
    except Exception as e:
        logger.error('cannot get gencfg group prj and geo: %s %s' % (type(e), e))
        return False

    return yasm_data_exists(prj, geo)


def init_logger(args):
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    stream = sys.stdout
    if args.monrun:
        stream = open(os.devnull, 'w')

    ch = logging.StreamHandler(stream)
    ch.setLevel(logging.DEBUG)
    ch.setFormatter(logging.Formatter('[%(asctime)s]\t%(filename)20s:%(funcName)-40s\t%(levelname)-8s\t%(message)s'))

    if (logger.hasHandlers()):
        logger.handlers.clear()
    logger.addHandler(ch)
    return logger


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('--monrun', default=False, action='store_true', help='писать результат проверки в формате monrun')
    parser.add_argument('-c', '--config', default='/etc/yandex-direct/direct-vhosts.yaml', help='путь до vhosts-конфига')
    parser.add_argument('-g', '--groups', default=[], nargs='+', help='Фильтр по gencfg группам')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()

    global logger
    logger = init_logger(args) # basicConfig не сработал
    logger.info('running with args: ' + str(args))

    with open(args.config) as f:
        conf = yaml.load(f)
    logger.info('loaded config: ' + str(conf))

    checks = []
    for vhost, vconf in conf['vhosts'].items():
        logger.info('processing vhost ' + vhost)

        gencfg_groups = vconf.get('endpoints', {}).get('gencfg-groups', [])
        if not gencfg_groups:
            continue

        for group in gencfg_groups:
            if args.groups and group not in args.groups:
                logger.info('skipping group %s as requested' % group)
                continue

            result = gencfg_group_yasm_data_exists(group)
            if not result:
                logger.warning('no yasm graphs for group %s, check tags in https://gencfg.yandex-team.ru/trunk/groups/%s and redeploy nanny services' % (group, group))

            logger.info('group %s yasm fetching result: %s' % (group, result))
            checks.append("" if result else group)

    # агрегируем проверки
    bad_groups = sorted(filter(bool, checks))
    descr = 'cannot fetch yasm graphs, run cmd for more info: dt-resmon-yasm -g ' + ' '.join(bad_groups) if bad_groups else 'OK'

    logger.info('aggregated result: ' + descr)
    if args.monrun:
        monrun_status = 2 if bad_groups else 0
        print('%d;%s' % (monrun_status, descr))
